## Quote to prevent word splitting/globbing, or split robustly with mapfile or read -a.

### Problematic code:

```sh
array=( $var )
```

### Correct code:

If the variable should be a single element:

```sh
array=( "$var" )
```

If it's multiple lines, each of which should be an element:

```sh
# For bash
mapfile -t array <<< "$var"

# For ksh
printf '%s\n' "$var" | while IFS="" read -r line; do array+=("$line"); done
```

If it's a line with multiple words (separated by spaces, other delimiters can be chosen with IFS), each of which should be an element:

```sh
# For bash
read -r -a array <<< "$var"

# For ksh
read -r -A array <<< "$var"
```
1. https://stackoverflow.com/questions/1469849

### Rationale:

You are expanding a variable unquoted in an array. This will invoke the shell's sloppy word splitting and glob expansion.

Instead, prefer explicitly splitting (or not splitting):

* If the variable should become a single array element, quote it.
* If you want to split into lines or words, use `mapfile`, `read -ra` and/or `while` loops as appropriate.

This prevents the shell from doing unwanted splitting and glob expansion, and therefore avoiding problems with data containing spaces or special characters.

### Exceptions:

If you have already taken care (through setting IFS and `set -f`) to have word splitting work the way you intend, you can ignore this warning.
[Source](https://github.com/koalaman/shellcheck/wiki/SC2206)

