
When a class implements the `IEquatable<T>` interface, it enters a contract that, in effect, states "I know how to compare two<br>instances of type T or any type derived from T for equality.". However if that class is derived, it is very unlikely that the base class will know how<br>to make a meaningful comparison. Therefore that implicit contract is now broken.

Alternatively `IEqualityComparer<T>` provides a safer interface and is used by collections or `Equals` could be made<br>`virtual`.

This rule raises an issue when a unsealed, `public` or `protected` class implements `IEquitable<T>` and the<br>`Equals` is neither `virtual` nor `abstract`.

## Noncompliant Code Example


    using System;
    
    namespace MyLibrary
    {
      class Base : IEquatable&lt;Base&gt; // Noncompliant
      {
        bool Equals(Base other)
        {
          if (other == null) { return false };
          // do comparison of base properties
        }
    
        override bool Equals(object other)  =&gt; Equals(other as Base);
      }
    
      class A : Base
      {
        bool Equals(A other)
        {
          if (other == null) { return false };
          // do comparison of A properties
          return base.Equals(other);
        }
    
        override bool Equals(object other)  =&gt; Equals(other as A);
      }
    
      class B : Base
      {
        bool Equals(B other)
        {
          if (other == null) { return false };
          // do comparison of B properties
         return base.Equals(other);
        }
    
        override bool Equals(object other)  =&gt; Equals(other as B);
      }
    
      static void Main() {
        A a = new A();
        B b = new B();
    
        Console.WriteLine(a.Equals(b)); // This calls the WRONG equals. This causes Base::Equals(Base)
        //  to be called which only compares the properties in Base and ignores the fact that
        // a and b are different types. In the working example A::Equals(Object) would have been
        // called and Equals would return false because it correctly recognizes that a and b are
        // different types. If a and b have the same base properties they will  be returned as equal.
      }
    }


## Compliant Solution


    using System;
    
    namespace MyLibrary
    {
        public sealed class Foo : IEquatable&lt;Foo&gt;
        {
            public bool Equals(Foo other)
            {
                // Your code here
            }
        }
    }


## See

[Inheritance and IEquatable do not mix](http://blog.mischel.com/2013/01/05/inheritance-and-iequatable-do-not-mix/)

[IEqualityComparer&lt;T&gt; Interface](https://msdn.microsoft.com/en-us/library/ms132151%28v=vs.110%29.aspx)
