
Using pseudorandom number generators (PRNGs) is security-sensitive. For example, it has led in the past to the following vulnerabilities:

- [CVE-2013-6386](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2013-6386)
- [CVE-2006-3419](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2006-3419)
- [CVE-2008-4102](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2008-4102)


When software generates predictable values in a context requiring unpredictability, it may be possible for an attacker to guess the next value that<br>will be generated, and use this guess to impersonate another user or access sensitive information.

As the `System.Random` class relies on a pseudorandom number generator, it should not be used for security-critical applications or for<br>protecting sensitive data. In such context, the `System.Cryptography.RandomNumberGenerator` class which relies on a cryptographically<br>strong random number generator (RNG) should be used in place.

## Ask Yourself Whether

- the code using the generated value requires it to be unpredictable. It is the case for all encryption mechanisms or when a secret value, such<br>  as a password, is hashed.
- the function you use generates a value which can be predicted (pseudo-random).
- the generated value is used multiple times.
- an attacker can access the generated value.


You are at risk if you answered yes to the first question and any of the following ones.

## Recommended Secure Coding Practices

- Only use random number generators which are [recommended by OWASP](https://www.owasp.org/index.php/Cryptographic_Storage_Cheat_Sheet#Rule_-_Use_strong_random_numbers) or any other<br>  trusted organization.
- Use the generated random values only once.
- You should not expose the generated random value. If you have to store it, make sure that the database or file is secure.


## Sensitive Code Example


    var random = new Random(); // Sensitive use of Random
    byte[] data = new byte[16];
    random.NextBytes(data);
    return BitConverter.ToString(data); // Check if this value is used for hashing or encryption


## Compliant Solution


    using System.Security.Cryptography;
    ...
    var randomGenerator = RandomNumberGenerator.Create(); // Compliant for security-sensitive use cases
    byte[] data = new byte[16];
    randomGenerator.GetBytes(data);
    return BitConverter.ToString(data);


## See

- [OWASP Top 10 2017 Category A3](https://www.owasp.org/index.php/Top_10-2017_A3-Sensitive_Data_Exposure) - Sensitive Data Exposure<br>
- [MITRE, CWE-338](http://cwe.mitre.org/data/definitions/338.html) - Use of Cryptographically Weak Pseudo-Random Number Generator<br>  (PRNG)
- [MITRE, CWE-330](http://cwe.mitre.org/data/definitions/330.html) - Use of Insufficiently Random Values
- [MITRE, CWE-326](http://cwe.mitre.org/data/definitions/326.html) - Inadequate Encryption Strength
- [CERT, MSC02-J.](https://www.securecoding.cert.org/confluence/x/mAFqAQ) - Generate strong random numbers
- [CERT, MSC30-C.](https://www.securecoding.cert.org/confluence/x/qw4) - Do not use the rand() function for generating pseudorandom<br>  numbers
- [CERT, MSC50-CPP.](https://www.securecoding.cert.org/confluence/x/WYIyAQ) - Do not use std::rand() for generating pseudorandom<br>  numbers
- Derived from FindSecBugs rule [Predictable Pseudo Random Number<br>  Generator](http://h3xstream.github.io/find-sec-bugs/bugs.htm#PREDICTABLE_RANDOM)

