
Controlling permissions is security-sensitive. It has led in the past to the following vulnerabilities:

- [CVE-2018-12999](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2018-12999)
- [CVE-2018-10285](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2018-10285)
- [CVE-2017-7455](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2017-7455)


Attackers can only damage what they have access to. Thus limiting their access is a good way to prevent them from wreaking havoc, but it has to be<br>done properly.

This rule flags code that controls the access to resources and actions or configures this access. The goal is to guide security code reviews.

## Ask Yourself Whether

- at least one accessed action or resource is security-sensitive.
- there is no access control in place or it does not cover all sensitive actions and resources.
- users have permissions they don't need.
- the access control is based on a user input or on some other unsafe data.
- permissions are difficult to remove or take a long time to be updated.


You are at risk if you answered yes to the first question and any of the following ones.

## Recommended Secure Coding Practices

The first step is to restrict all sensitive actions to authenticated users.

Each user should have the lowest privileges possible. The access control granularity should match the sensitivity of each resource or action. The<br>more sensitive it is, the less people should have access to it.

Do not base the access control on a user input or on a value which might have been tampered with. For example, the developer should not read a<br>user's permissions from an HTTP cookie as it can be modified client-side.

Check that the access to each action and resource is properly restricted.

Enable administrators to swiftly remove permissions when necessary. This enables them to reduce the time an attacker can have access to your<br>systems when a breach occurs.

Log and monitor refused access requests as they can reveal an attack.

## Sensitive Code Example


    using System.Threading;
    using System.Security.Permissions;
    using System.Security.Principal;
    using System.IdentityModel.Tokens;
    
    class SecurityPrincipalDemo
    {
        class MyIdentity : IIdentity // Sensitive, custom IIdentity implementations should be reviewed
        {
            // ...
        }
    
        class MyPrincipal : IPrincipal // Sensitive, custom IPrincipal implementations should be reviewed
        {
            // ...
        }
        [System.Security.Permissions.PrincipalPermission(SecurityAction.Demand, Role = "Administrators")] // Sensitive. The access restrictions enforced by this attribute should be reviewed.
        static void CheckAdministrator()
        {
            WindowsIdentity MyIdentity = WindowsIdentity.GetCurrent(); // Sensitive
            HttpContext.User = ...; // Sensitive: review all reference (set and get) to System.Web HttpContext.User
            AppDomain domain = AppDomain.CurrentDomain;
            domain.SetPrincipalPolicy(PrincipalPolicy.WindowsPrincipal); // Sensitive
            MyIdentity identity = new MyIdentity(); // Sensitive
            MyPrincipal MyPrincipal = new MyPrincipal(MyIdentity); // Sensitive
            Thread.CurrentPrincipal = MyPrincipal; // Sensitive
            domain.SetThreadPrincipal(MyPrincipal); // Sensitive
    
            // All instantiation of PrincipalPermission should be reviewed.
            PrincipalPermission principalPerm = new PrincipalPermission(null, "Administrators"); // Sensitive
            principalPerm.Demand();
    
            SecurityTokenHandler handler = ...;
            // Sensitive: this creates an identity.
            ReadOnlyCollection&lt;ClaimsIdentity&gt; identities = handler.ValidateToken(…);
        }
    
         // Sensitive: review how this function uses the identity and principal.
        void modifyPrincipal(MyIdentity identity, MyPrincipal principal)
        {
            // ...
        }
    }


## See

- [OWASP Top 10 2017 Category A5](https://www.owasp.org/index.php/Top_10-2017_A5-Broken_Access_Control) - Boken Access Control
- [SANS Top 25](https://www.sans.org/top25-software-errors/#cat3) - Porous Defenses

