
Catching `System.Exception` seems like an efficient way to handle multiple possible exceptions. Unfortunately, it traps all exception types, including the ones that were not intended to be caught. To prevent any misunderstandings, the exception filters should be used. Alternatively each exception type should be in a separate `catch` block.

## Noncompliant Code Example


    try
    {
      // do something that might throw a FileNotFoundException or IOException
    }
    catch (Exception e) // Noncompliant
    {
      // log exception ...
    }


## Compliant Solution


    try
    {
      // do something
    }
    catch (Exception e) when (e is FileNotFoundException || e is IOException)
    {
      // do something
    }


## Exceptions

The final option is to catch `System.Exception` and `throw` it in the last statement in the `catch` block. This is the least-preferred option, as it is an old-style code, which also suffers from performance penalty compared to exception filters.


    try
    {
      // do something
    }
    catch (Exception e)
    {
      if (e is FileNotFoundException || e is IOException)
      {
        // do something
      }
      else
      {
        throw;
      }
    }


## See

- [MITRE, CWE-396](http://cwe.mitre.org/data/definitions/396.html) - Declaration of Catch for Generic Exception

