
Configuring loggers is security-sensitive. It has led in the past to the following vulnerabilities:

- [CVE-2018-0285](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2018-0285)
- [CVE-2000-1127](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2000-1127)
- [CVE-2017-15113](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2017-15113)
- [CVE-2015-5742](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2015-5742)


Logs are useful before, during and after a security incident.

- Attackers will most of the time start their nefarious work by probing the system for vulnerabilities. Monitoring this activity and stopping it<br>  is the first step to prevent an attack from ever happening.
- In case of a successful attack, logs should contain enough information to understand what damage an attacker may have inflicted.


Logs are also a target for attackers because they might contain sensitive information. Configuring loggers has an impact on the type of information logged and how they are logged.

This rule flags for review code that initiates loggers configuration. The goal is to guide security code reviews.

## Ask Yourself Whether

- unauthorized users might have access to the logs, either because they are stored in an insecure location or because the application gives<br>  access to them.
- the logs contain sensitive information on a production server. This can happen when the logger is in debug mode.
- the log can grow without limit. This can happen when additional information is written into logs every time a user performs an action and the<br>  user can perform the action as many times as he/she wants.
- the logs do not contain enough information to understand the damage an attacker might have inflicted. The loggers mode (info, warn, error)<br>  might filter out important information. They might not print contextual information like the precise time of events or the server hostname.
- the logs are only stored locally instead of being backuped or replicated.


You are at risk if you answered yes to any of those questions.

## Recommended Secure Coding Practices

- Check that your production deployment doesn't have its loggers in "debug" mode as it might write sensitive information in logs.
- Production logs should be stored in a secure location which is only accessible to system administrators.
- Configure the loggers to display all warnings, info and error messages. Write relevant information such as the precise time of events and the<br>  hostname.
- Choose log format which is easy to parse and process automatically. It is important to process logs rapidly in case of an attack so that the<br>  impact is known and limited.
- Check that the permissions of the log files are correct. If you index the logs in some other service, make sure that the transfer and the<br>  service are secure too.
- Add limits to the size of the logs and make sure that no user can fill the disk with logs. This can happen even when the user does not control<br>  the logged information. An attacker could just repeat a logged action many times.


Remember that configuring loggers properly doesn't make them bullet-proof. Here is a list of recommendations explaining on how to use your logs:

- Don't log any sensitive information. This obviously includes passwords and credit card numbers but also any personal information such as user<br>  names, locations, etc... Usually any information which is protected by law is good candidate for removal.
- Sanitize all user inputs before writing them in the logs. This includes checking its size, content, encoding, syntax, etc... As for any user<br>  input, validate using whitelists whenever possible. Enabling users to write what they want in your logs can have many impacts. It could for example<br>  use all your storage space or compromise your log indexing service.
- Log enough information to monitor suspicious activities and evaluate the impact an attacker might have on your systems. Register events such as<br>  failed logins, successful logins, server side input validation failures, access denials and any important transaction.
- Monitor the logs for any suspicious activity.


## Sensitive Code Example

**.Net Core**: configure programmatically


    using System;
    using System.Collections;
    using System.Collections.Generic;
    using Microsoft.AspNetCore.Builder;
    using Microsoft.AspNetCore.Hosting;
    using Microsoft.Extensions.Configuration;
    using Microsoft.Extensions.DependencyInjection;
    using Microsoft.Extensions.Logging;
    using Microsoft.Extensions.Options;
    using Microsoft.AspNetCore;
    
    namespace MvcApp
    {
        public class ProgramLogging
        {
            public static IWebHostBuilder CreateWebHostBuilder(string[] args) =&gt;
                WebHost.CreateDefaultBuilder(args)
                    .ConfigureLogging((hostingContext, logging) =&gt; // Sensitive
                    {
                        // ...
                    })
                    .UseStartup&lt;StartupLogging&gt;();
        }
    
        public class StartupLogging
        {
            public void ConfigureServices(IServiceCollection services)
            {
                services.AddLogging(logging =&gt; // Sensitive
                {
                    // ...
                });
            }
    
            public void Configure(IApplicationBuilder app, IHostingEnvironment env, ILoggerFactory loggerFactory)
            {
                IConfiguration config = null;
                LogLevel level = LogLevel.Critical;
                Boolean includeScopes = false;
                Func&lt;string,Microsoft.Extensions.Logging.LogLevel,bool&gt; filter = null;
                Microsoft.Extensions.Logging.Console.IConsoleLoggerSettings consoleSettings = null;
                Microsoft.Extensions.Logging.AzureAppServices.AzureAppServicesDiagnosticsSettings azureSettings = null;
                Microsoft.Extensions.Logging.EventLog.EventLogSettings eventLogSettings = null;
    
                // An issue will be raised for each call to an ILoggerFactory extension methods adding loggers.
                loggerFactory.AddAzureWebAppDiagnostics(); // Sensitive
                loggerFactory.AddAzureWebAppDiagnostics(azureSettings); // Sensitive
                loggerFactory.AddConsole(); // Sensitive
                loggerFactory.AddConsole(level); // Sensitive
                loggerFactory.AddConsole(level, includeScopes); // Sensitive
                loggerFactory.AddConsole(filter); // Sensitive
                loggerFactory.AddConsole(filter, includeScopes); // Sensitive
                loggerFactory.AddConsole(config); // Sensitive
                loggerFactory.AddConsole(consoleSettings); // Sensitive
                loggerFactory.AddDebug(); // Sensitive
                loggerFactory.AddDebug(level); // Sensitive
                loggerFactory.AddDebug(filter); // Sensitive
                loggerFactory.AddEventLog(); // Sensitive
                loggerFactory.AddEventLog(eventLogSettings); // Sensitive
                loggerFactory.AddEventLog(level); // Sensitive
                loggerFactory.AddEventSourceLogger(); // Sensitive
    
                IEnumerable&lt;ILoggerProvider&gt; providers = null;
                LoggerFilterOptions filterOptions1 = null;
                IOptionsMonitor&lt;LoggerFilterOptions&gt; filterOptions2 = null;
    
                LoggerFactory factory = new LoggerFactory(); // Sensitive
                new LoggerFactory(providers); // Sensitive
                new LoggerFactory(providers, filterOptions1); // Sensitive
                new LoggerFactory(providers, filterOptions2); // Sensitive
            }
        }
    }


**Log4Net**


    using System;
    using System.IO;
    using System.Xml;
    using log4net.Appender;
    using log4net.Config;
    using log4net.Repository;
    
    namespace Logging
    {
        class Log4netLogging
        {
            void Foo(ILoggerRepository repository, XmlElement element, FileInfo configFile, Uri configUri, Stream configStream,
            IAppender appender, params IAppender[] appenders) {
                log4net.Config.XmlConfigurator.Configure(repository); // Sensitive
                log4net.Config.XmlConfigurator.Configure(repository, element); // Sensitive
                log4net.Config.XmlConfigurator.Configure(repository, configFile); // Sensitive
                log4net.Config.XmlConfigurator.Configure(repository, configUri); // Sensitive
                log4net.Config.XmlConfigurator.Configure(repository, configStream); // Sensitive
                log4net.Config.XmlConfigurator.ConfigureAndWatch(repository, configFile); // Sensitive
    
                log4net.Config.DOMConfigurator.Configure(); // Sensitive
                log4net.Config.DOMConfigurator.Configure(repository); // Sensitive
                log4net.Config.DOMConfigurator.Configure(element); // Sensitive
                log4net.Config.DOMConfigurator.Configure(repository, element); // Sensitive
                log4net.Config.DOMConfigurator.Configure(configFile); // Sensitive
                log4net.Config.DOMConfigurator.Configure(repository, configFile); // Sensitive
                log4net.Config.DOMConfigurator.Configure(configStream); // Sensitive
                log4net.Config.DOMConfigurator.Configure(repository, configStream); // Sensitive
                log4net.Config.DOMConfigurator.ConfigureAndWatch(configFile); // Sensitive
                log4net.Config.DOMConfigurator.ConfigureAndWatch(repository, configFile); // Sensitive
    
                log4net.Config.BasicConfigurator.Configure(); // Sensitive
                log4net.Config.BasicConfigurator.Configure(appender); // Sensitive
                log4net.Config.BasicConfigurator.Configure(appenders); // Sensitive
                log4net.Config.BasicConfigurator.Configure(repository); // Sensitive
                log4net.Config.BasicConfigurator.Configure(repository, appender); // Sensitive
                log4net.Config.BasicConfigurator.Configure(repository, appenders); // Sensitive
            }
        }
    }


**NLog**: configure programmatically


    namespace Logging
    {
        class NLogLogging
        {
            void Foo(NLog.Config.LoggingConfiguration config) {
                NLog.LogManager.Configuration = config; // Sensitive, this changes the logging configuration.
            }
        }
    }


**Serilog**


    namespace Logging
    {
        class SerilogLogging
        {
            void Foo() {
                new Serilog.LoggerConfiguration(); // Sensitive
            }
        }
    }


## See

- [OWASP Top 10 2017 Category A3](https://www.owasp.org/index.php/Top_10-2017_A3-Sensitive_Data_Exposure) - Sensitive Data Exposure<br>
- [OWASP Top 10 2017 Category A10](https://www.owasp.org/index.php/Top_10-2017_A10-Insufficient_Logging%26Monitoring) - Insufficient<br>  Logging & Monitoring
- [MITRE, CWE-532](https://cwe.mitre.org/data/definitions/532.html) - Information Exposure Through Log Files
- [MITRE, CWE-117](https://cwe.mitre.org/data/definitions/117.html) - Improper Output Neutralization for Logs
- [MITRE, CWE-778](https://cwe.mitre.org/data/definitions/778.html) - Insufficient Logging
- [SANS Top 25](https://www.sans.org/top25-software-errors/#cat3) - Porous Defenses

