package com.codacy.tracing.core

import scala.concurrent.{ExecutionContext, Future}

/**
  * A Span pans are created by calling the buildSpan(operationName) method on Kamon’s companion object as shown below:
  */
trait SpanBuilder {

  /**
    * Starts a span.
    * Should only be used when full control of start and end are needed.
    * Returns a `Span` with the method `finish`.
    */
  def start(): Span

  /**
    * Runs f and recording it as a span
    */
  def withSpan[T](f: => T): T = {
    val span = start()

    try {
      f
    } finally {
      span.finish()
    }
  }

  /**
    * The span registers 'now' up to the moment that the given `future` completes.
    *
    * This method may measure more than is obvious.
    * It measures:
    * * the evaluation of the (by name) parameter `future`
    * * in case the future is not yet completed: the delay until the constructed Future is scheduled in the
    *   given `ExecutionContext`
    * * in case the future is not yet completed: the actual execution of the Future
    * * the time it takes to schedule finishing the span
    *
    * Source: https://github.com/erikvanoosten/metrics-scala/blob/c20187a1ed09f50a523b310e3a7ffa992564afb3/metrics-scala/src/main/scala/nl/grons/metrics4/scala/Timer.scala#L60
    */
  def withSpan[A](future: => Future[A])(implicit context: ExecutionContext): Future[A] = {
    val span = start()
    val f = try {
      future
    } catch {
      case ex: Throwable =>
        span.finish()
        throw ex
    }

    // Using `map` and `recoverWith` instead of `onComplete` to grantee that `ctx.stop()` is executed
    f.map { v =>
        span.finish()
        v
      }
      .recoverWith {
        case v =>
          span.finish()
          Future.failed(v)
      }
  }

}

/**
  * Representation for a Span that was started.
  */
private[tracing] trait Span {

  /**
    * Finishes and registers the `Span`.
    */
  def finish(): Unit

}
