package com.commercetools.graphql.api.client;

import com.commercetools.graphql.api.types.CustomerSignInDraft;
import com.netflix.graphql.dgs.client.codegen.GraphQLQuery;
import java.lang.Override;
import java.lang.String;
import java.util.HashSet;
import java.util.Set;

/**
 * Retrieves the authenticated customer (a customer that matches the given email/password pair).
 *
 * There may be carts and orders created before the sign in that should be assigned to the customer account. With the `anonymousCartId`, a single anonymous cart can be assigned. With the `anonymousId`, all orders and carts that have this `anonymousId` set will be assigned to the customer.
 * If both `anonymousCartId` and `anonymousId` are given, the anonymous cart must have the `anonymousId`.
 *
 * Additionally, there might also exist one or more active customer carts from an earlier session. On customer sign in there are several ways how to proceed with this cart and the cart referenced by the `anonymousCartId`.
 *
 * * If the customer does not have a cart yet, the anonymous cart becomes the customer's cart.
 * * If the customer already has one or more carts, the content of the anonymous cart will be copied to the customer's active cart that has been modified most recently.
 *
 *   In this case the `CartState` of the anonymous cart gets changed to `Merged` while the customer's cart remains the `Active` cart.
 *
 *   If a `LineItem` in the anonymous cart matches an existing line item, or a `CustomLineItem` matches an existing custom line item in the customer's cart, the maximum quantity of both line items is used as the new quantity.
 *
 *   `ItemShippingDetails` are copied from the item with the highest quantity.
 *
 *   If `itemShippingAddresses` are different in the two carts, the resulting cart contains the addresses of both the customer cart and the anonymous cart.
 *
 *   Note, that it is not possible to merge carts that differ in their currency (set during creation of the cart).
 *
 * If a cart is is returned as part of the `CustomerSignInResult`, it has been recalculated (it will have up-to-date prices, taxes and discounts, and invalid line items have been removed).
 */
public class CustomerSignInGraphQLQuery extends GraphQLQuery {
  public CustomerSignInGraphQLQuery(CustomerSignInDraft draft, String storeKey, String queryName,
      Set<String> fieldsSet) {
    super("mutation", queryName);
    if (draft != null || fieldsSet.contains("draft")) {
        getInput().put("draft", draft);
    }if (storeKey != null || fieldsSet.contains("storeKey")) {
        getInput().put("storeKey", storeKey);
    }
  }

  public CustomerSignInGraphQLQuery() {
    super("mutation");
  }

  @Override
  public String getOperationName() {
    return "customerSignIn";
  }

  public static Builder newRequest() {
    return new Builder();
  }

  public static class Builder {
    private Set<String> fieldsSet = new HashSet<>();

    private CustomerSignInDraft draft;

    private String storeKey;

    private String queryName;

    public CustomerSignInGraphQLQuery build() {
      return new CustomerSignInGraphQLQuery(draft, storeKey, queryName, fieldsSet);
               
    }

    public Builder draft(CustomerSignInDraft draft) {
      this.draft = draft;
      this.fieldsSet.add("draft");
      return this;
    }

    /**
     * The mutation is only performed if the resource is part of the store. Can be used with store-specific OAuth permissions.
     */
    public Builder storeKey(String storeKey) {
      this.storeKey = storeKey;
      this.fieldsSet.add("storeKey");
      return this;
    }

    public Builder queryName(String queryName) {
      this.queryName = queryName;
      return this;
    }
  }
}
