package com.commercetools.graphql.api.types;

import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.util.Objects;

/**
 * A key-value pair representing the field name and value of one single custom field.
 *
 * The value of this custom field consists of escaped JSON based on the FieldDefinition of the Type.
 *
 *
 * Examples for `value`:
 *
 * * FieldType `String`: `"\"This is a string\""`
 * * FieldType `DateTimeType`: `"\"2018-10-12T14:00:00.000Z\""`
 * * FieldType `Number`: `"4"`
 * * FieldType `Set` with an elementType of `String`: `"[\"This is a string\", \"This is another string\"]"`
 * * FieldType `Reference`: `"{\"id\": \"b911b62d-353a-4388-93ee-8d488d9af962\", \"typeId\": \"product\"}"`
 */
public class CustomFieldInput {
  private String name;

  /**
   * The value of this custom field consists of escaped JSON based on the FieldDefinition of the Type.
   *
   *
   * Examples for `value`:
   *
   * * FieldType `String`: `"\"This is a string\""`
   * * FieldType `DateTimeType`: `"\"2018-10-12T14:00:00.000Z\""`
   * * FieldType `Number`: `"4"`
   * * FieldType `Set` with an elementType of `String`: `"[\"This is a string\", \"This is another string\"]"`
   * * FieldType `Reference`: `"{\"id\": \"b911b62d-353a-4388-93ee-8d488d9af962\", \"typeId\": \"product\"}"`
   */
  private String value;

  public CustomFieldInput() {
  }

  public CustomFieldInput(String name, String value) {
    this.name = name;
    this.value = value;
  }

  public String getName() {
    return name;
  }

  public void setName(String name) {
    this.name = name;
  }

  /**
   * The value of this custom field consists of escaped JSON based on the FieldDefinition of the Type.
   *
   *
   * Examples for `value`:
   *
   * * FieldType `String`: `"\"This is a string\""`
   * * FieldType `DateTimeType`: `"\"2018-10-12T14:00:00.000Z\""`
   * * FieldType `Number`: `"4"`
   * * FieldType `Set` with an elementType of `String`: `"[\"This is a string\", \"This is another string\"]"`
   * * FieldType `Reference`: `"{\"id\": \"b911b62d-353a-4388-93ee-8d488d9af962\", \"typeId\": \"product\"}"`
   */
  public String getValue() {
    return value;
  }

  public void setValue(String value) {
    this.value = value;
  }

  @Override
  public String toString() {
    return "CustomFieldInput{name='" + name + "', value='" + value + "'}";
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    CustomFieldInput that = (CustomFieldInput) o;
    return Objects.equals(name, that.name) &&
        Objects.equals(value, that.value);
  }

  @Override
  public int hashCode() {
    return Objects.hash(name, value);
  }

  public static Builder newBuilder() {
    return new Builder();
  }

  public static class Builder {
    private String name;

    /**
     * The value of this custom field consists of escaped JSON based on the FieldDefinition of the Type.
     *
     *
     * Examples for `value`:
     *
     * * FieldType `String`: `"\"This is a string\""`
     * * FieldType `DateTimeType`: `"\"2018-10-12T14:00:00.000Z\""`
     * * FieldType `Number`: `"4"`
     * * FieldType `Set` with an elementType of `String`: `"[\"This is a string\", \"This is another string\"]"`
     * * FieldType `Reference`: `"{\"id\": \"b911b62d-353a-4388-93ee-8d488d9af962\", \"typeId\": \"product\"}"`
     */
    private String value;

    public CustomFieldInput build() {
      CustomFieldInput result = new CustomFieldInput();
      result.name = this.name;
      result.value = this.value;
      return result;
    }

    public Builder name(String name) {
      this.name = name;
      return this;
    }

    /**
     * The value of this custom field consists of escaped JSON based on the FieldDefinition of the Type.
     *
     *
     * Examples for `value`:
     *
     * * FieldType `String`: `"\"This is a string\""`
     * * FieldType `DateTimeType`: `"\"2018-10-12T14:00:00.000Z\""`
     * * FieldType `Number`: `"4"`
     * * FieldType `Set` with an elementType of `String`: `"[\"This is a string\", \"This is another string\"]"`
     * * FieldType `Reference`: `"{\"id\": \"b911b62d-353a-4388-93ee-8d488d9af962\", \"typeId\": \"product\"}"`
     */
    public Builder value(String value) {
      this.value = value;
      return this;
    }
  }
}
