package com.datadog.android.rum.model

import com.datadog.android.core.`internal`.utils.JsonSerializer
import com.google.gson.JsonArray
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import com.google.gson.JsonPrimitive
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Any
import kotlin.Array
import kotlin.Boolean
import kotlin.Long
import kotlin.Number
import kotlin.String
import kotlin.collections.ArrayList
import kotlin.collections.List
import kotlin.collections.MutableMap
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Schema of common properties of RUM events
 * Schema of all properties of events that can have parent actions
 * @param date Start of the event in ms from epoch
 * @param application Application properties
 * @param service The service name for this application
 * @param version The version for this application
 * @param session Session properties
 * @param source The source of this event
 * @param view View properties
 * @param usr User properties
 * @param connectivity Device connectivity properties
 * @param display Display properties
 * @param synthetics Synthetics properties
 * @param ciTest CI Visibility properties
 * @param os Operating system properties
 * @param device Device properties
 * @param dd Internal properties
 * Internal properties
 * @param context User provided context
 * @param action Action properties
 * @param resource Resource properties
 */
public data class ResourceEvent(
    public val date: Long,
    public val application: Application,
    public val service: String? = null,
    public val version: String? = null,
    public val session: ResourceEventSession,
    public val source: Source? = null,
    public val view: View,
    public val usr: Usr? = null,
    public val connectivity: Connectivity? = null,
    public val display: Display? = null,
    public val synthetics: Synthetics? = null,
    public val ciTest: CiTest? = null,
    public val os: Os? = null,
    public val device: Device? = null,
    public val dd: Dd,
    public val context: Context? = null,
    public val action: Action? = null,
    public val resource: Resource,
) {
    public val type: String = "resource"

    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.addProperty("date", date)
        json.add("application", application.toJson())
        service?.let { serviceNonNull ->
            json.addProperty("service", serviceNonNull)
        }
        version?.let { versionNonNull ->
            json.addProperty("version", versionNonNull)
        }
        json.add("session", session.toJson())
        source?.let { sourceNonNull ->
            json.add("source", sourceNonNull.toJson())
        }
        json.add("view", view.toJson())
        usr?.let { usrNonNull ->
            json.add("usr", usrNonNull.toJson())
        }
        connectivity?.let { connectivityNonNull ->
            json.add("connectivity", connectivityNonNull.toJson())
        }
        display?.let { displayNonNull ->
            json.add("display", displayNonNull.toJson())
        }
        synthetics?.let { syntheticsNonNull ->
            json.add("synthetics", syntheticsNonNull.toJson())
        }
        ciTest?.let { ciTestNonNull ->
            json.add("ci_test", ciTestNonNull.toJson())
        }
        os?.let { osNonNull ->
            json.add("os", osNonNull.toJson())
        }
        device?.let { deviceNonNull ->
            json.add("device", deviceNonNull.toJson())
        }
        json.add("_dd", dd.toJson())
        context?.let { contextNonNull ->
            json.add("context", contextNonNull.toJson())
        }
        action?.let { actionNonNull ->
            json.add("action", actionNonNull.toJson())
        }
        json.addProperty("type", type)
        json.add("resource", resource.toJson())
        return json
    }

    public companion object {
        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): ResourceEvent {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type ResourceEvent",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): ResourceEvent {
            try {
                val date = jsonObject.get("date").asLong
                val application = jsonObject.get("application").asJsonObject.let {
                    Application.fromJsonObject(it)
                }
                val service = jsonObject.get("service")?.asString
                val version = jsonObject.get("version")?.asString
                val session = jsonObject.get("session").asJsonObject.let {
                    ResourceEventSession.fromJsonObject(it)
                }
                val source = jsonObject.get("source")?.asString?.let {
                    Source.fromJson(it)
                }
                val view = jsonObject.get("view").asJsonObject.let {
                    View.fromJsonObject(it)
                }
                val usr = jsonObject.get("usr")?.asJsonObject?.let {
                    Usr.fromJsonObject(it)
                }
                val connectivity = jsonObject.get("connectivity")?.asJsonObject?.let {
                    Connectivity.fromJsonObject(it)
                }
                val display = jsonObject.get("display")?.asJsonObject?.let {
                    Display.fromJsonObject(it)
                }
                val synthetics = jsonObject.get("synthetics")?.asJsonObject?.let {
                    Synthetics.fromJsonObject(it)
                }
                val ciTest = jsonObject.get("ci_test")?.asJsonObject?.let {
                    CiTest.fromJsonObject(it)
                }
                val os = jsonObject.get("os")?.asJsonObject?.let {
                    Os.fromJsonObject(it)
                }
                val device = jsonObject.get("device")?.asJsonObject?.let {
                    Device.fromJsonObject(it)
                }
                val dd = jsonObject.get("_dd").asJsonObject.let {
                    Dd.fromJsonObject(it)
                }
                val context = jsonObject.get("context")?.asJsonObject?.let {
                    Context.fromJsonObject(it)
                }
                val action = jsonObject.get("action")?.asJsonObject?.let {
                    Action.fromJsonObject(it)
                }
                val resource = jsonObject.get("resource").asJsonObject.let {
                    Resource.fromJsonObject(it)
                }
                return ResourceEvent(date, application, service, version, session, source, view,
                        usr, connectivity, display, synthetics, ciTest, os, device, dd, context,
                        action, resource)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type ResourceEvent",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type ResourceEvent",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type ResourceEvent",
                    e
                )
            }
        }
    }

    /**
     * Application properties
     * @param id UUID of the application
     */
    public data class Application(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Application {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Application {
                try {
                    val id = jsonObject.get("id").asString
                    return Application(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }
        }
    }

    /**
     * Session properties
     * @param id UUID of the session
     * @param type Type of the session
     * @param hasReplay Whether this session has a replay
     */
    public data class ResourceEventSession(
        public val id: String,
        public val type: ResourceEventSessionType,
        public val hasReplay: Boolean? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            json.add("type", type.toJson())
            hasReplay?.let { hasReplayNonNull ->
                json.addProperty("has_replay", hasReplayNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): ResourceEventSession {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ResourceEventSession",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): ResourceEventSession {
                try {
                    val id = jsonObject.get("id").asString
                    val type = ResourceEventSessionType.fromJson(jsonObject.get("type").asString)
                    val hasReplay = jsonObject.get("has_replay")?.asBoolean
                    return ResourceEventSession(id, type, hasReplay)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ResourceEventSession",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type ResourceEventSession",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type ResourceEventSession",
                        e
                    )
                }
            }
        }
    }

    /**
     * View properties
     * @param id UUID of the view
     * @param referrer URL that linked to the initial view of the page
     * @param url URL of the view
     * @param name User defined name of the view
     */
    public data class View(
        public val id: String,
        public var referrer: String? = null,
        public var url: String,
        public var name: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            referrer?.let { referrerNonNull ->
                json.addProperty("referrer", referrerNonNull)
            }
            json.addProperty("url", url)
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): View {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): View {
                try {
                    val id = jsonObject.get("id").asString
                    val referrer = jsonObject.get("referrer")?.asString
                    val url = jsonObject.get("url").asString
                    val name = jsonObject.get("name")?.asString
                    return View(id, referrer, url, name)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }
        }
    }

    /**
     * User properties
     * @param id Identifier of the user
     * @param name Name of the user
     * @param email Email of the user
     */
    public data class Usr(
        public val id: String? = null,
        public val name: String? = null,
        public val email: String? = null,
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            email?.let { emailNonNull ->
                json.addProperty("email", emailNonNull)
            }
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.add(k, JsonSerializer.toJsonElement(v))
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("id", "name", "email")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Usr {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Usr {
                try {
                    val id = jsonObject.get("id")?.asString
                    val name = jsonObject.get("name")?.asString
                    val email = jsonObject.get("email")?.asString
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value
                        }
                    }
                    return Usr(id, name, email, additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                }
            }
        }
    }

    /**
     * Device connectivity properties
     * @param status Status of the device connectivity
     * @param interfaces The list of available network interfaces
     * @param cellular Cellular connectivity properties
     */
    public data class Connectivity(
        public val status: Status,
        public val interfaces: List<Interface>,
        public val cellular: Cellular? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("status", status.toJson())
            val interfacesArray = JsonArray(interfaces.size)
            interfaces.forEach { interfacesArray.add(it.toJson()) }
            json.add("interfaces", interfacesArray)
            cellular?.let { cellularNonNull ->
                json.add("cellular", cellularNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Connectivity {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Connectivity {
                try {
                    val status = Status.fromJson(jsonObject.get("status").asString)
                    val interfaces = jsonObject.get("interfaces").asJsonArray.let { jsonArray ->
                        val collection = ArrayList<Interface>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(Interface.fromJson(it.asString))
                        }
                        collection
                    }
                    val cellular = jsonObject.get("cellular")?.asJsonObject?.let {
                        Cellular.fromJsonObject(it)
                    }
                    return Connectivity(status, interfaces, cellular)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                }
            }
        }
    }

    /**
     * Display properties
     * @param viewport The viewport represents the rectangular area that is currently being viewed.
     * Content outside the viewport is not visible onscreen until scrolled into view.
     */
    public data class Display(
        public val viewport: Viewport? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            viewport?.let { viewportNonNull ->
                json.add("viewport", viewportNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Display {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Display {
                try {
                    val viewport = jsonObject.get("viewport")?.asJsonObject?.let {
                        Viewport.fromJsonObject(it)
                    }
                    return Display(viewport)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                }
            }
        }
    }

    /**
     * Synthetics properties
     * @param testId The identifier of the current Synthetics test
     * @param resultId The identifier of the current Synthetics test results
     * @param injected Whether the event comes from a SDK instance injected by Synthetics
     */
    public data class Synthetics(
        public val testId: String,
        public val resultId: String,
        public val injected: Boolean? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("test_id", testId)
            json.addProperty("result_id", resultId)
            injected?.let { injectedNonNull ->
                json.addProperty("injected", injectedNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Synthetics {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Synthetics {
                try {
                    val testId = jsonObject.get("test_id").asString
                    val resultId = jsonObject.get("result_id").asString
                    val injected = jsonObject.get("injected")?.asBoolean
                    return Synthetics(testId, resultId, injected)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                }
            }
        }
    }

    /**
     * CI Visibility properties
     * @param testExecutionId The identifier of the current CI Visibility test execution
     */
    public data class CiTest(
        public val testExecutionId: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("test_execution_id", testExecutionId)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): CiTest {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): CiTest {
                try {
                    val testExecutionId = jsonObject.get("test_execution_id").asString
                    return CiTest(testExecutionId)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                }
            }
        }
    }

    /**
     * Operating system properties
     * @param name Operating system name, e.g. Android, iOS
     * @param version Full operating system version, e.g. 8.1.1
     * @param versionMajor Major operating system version, e.g. 8
     */
    public data class Os(
        public val name: String,
        public val version: String,
        public val versionMajor: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("name", name)
            json.addProperty("version", version)
            json.addProperty("version_major", versionMajor)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Os {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Os {
                try {
                    val name = jsonObject.get("name").asString
                    val version = jsonObject.get("version").asString
                    val versionMajor = jsonObject.get("version_major").asString
                    return Os(name, version, versionMajor)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }
        }
    }

    /**
     * Device properties
     * @param type Device type info
     * @param name Device marketing name, e.g. Xiaomi Redmi Note 8 Pro, Pixel 5, etc.
     * @param model Device SKU model, e.g. Samsung SM-988GN, etc. Quite often name and model can be
     * the same.
     * @param brand Device marketing brand, e.g. Apple, OPPO, Xiaomi, etc.
     * @param architecture The CPU architecture of the device that is reporting the error
     */
    public data class Device(
        public val type: DeviceType,
        public val name: String? = null,
        public val model: String? = null,
        public val brand: String? = null,
        public val architecture: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("type", type.toJson())
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            model?.let { modelNonNull ->
                json.addProperty("model", modelNonNull)
            }
            brand?.let { brandNonNull ->
                json.addProperty("brand", brandNonNull)
            }
            architecture?.let { architectureNonNull ->
                json.addProperty("architecture", architectureNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Device {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Device {
                try {
                    val type = DeviceType.fromJson(jsonObject.get("type").asString)
                    val name = jsonObject.get("name")?.asString
                    val model = jsonObject.get("model")?.asString
                    val brand = jsonObject.get("brand")?.asString
                    val architecture = jsonObject.get("architecture")?.asString
                    return Device(type, name, model, brand, architecture)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }
        }
    }

    /**
     * Internal properties
     * Internal properties
     * @param session Session-related internal properties
     * @param browserSdkVersion Browser SDK version
     * @param spanId span identifier in decimal format
     * @param traceId trace identifier in decimal format
     * @param rulePsr trace sample rate in decimal format
     * @param discarded Whether the resource should be discarded or indexed
     */
    public data class Dd(
        public val session: DdSession? = null,
        public val browserSdkVersion: String? = null,
        public val spanId: String? = null,
        public val traceId: String? = null,
        public val rulePsr: Number? = null,
        public val discarded: Boolean? = null,
    ) {
        public val formatVersion: Long = 2L

        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("format_version", formatVersion)
            session?.let { sessionNonNull ->
                json.add("session", sessionNonNull.toJson())
            }
            browserSdkVersion?.let { browserSdkVersionNonNull ->
                json.addProperty("browser_sdk_version", browserSdkVersionNonNull)
            }
            spanId?.let { spanIdNonNull ->
                json.addProperty("span_id", spanIdNonNull)
            }
            traceId?.let { traceIdNonNull ->
                json.addProperty("trace_id", traceIdNonNull)
            }
            rulePsr?.let { rulePsrNonNull ->
                json.addProperty("rule_psr", rulePsrNonNull)
            }
            discarded?.let { discardedNonNull ->
                json.addProperty("discarded", discardedNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Dd {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Dd {
                try {
                    val session = jsonObject.get("session")?.asJsonObject?.let {
                        DdSession.fromJsonObject(it)
                    }
                    val browserSdkVersion = jsonObject.get("browser_sdk_version")?.asString
                    val spanId = jsonObject.get("span_id")?.asString
                    val traceId = jsonObject.get("trace_id")?.asString
                    val rulePsr = jsonObject.get("rule_psr")?.asNumber
                    val discarded = jsonObject.get("discarded")?.asBoolean
                    return Dd(session, browserSdkVersion, spanId, traceId, rulePsr, discarded)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }
        }
    }

    /**
     * User provided context
     */
    public data class Context(
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            additionalProperties.forEach { (k, v) ->
                json.add(k, JsonSerializer.toJsonElement(v))
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Context {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Context {
                try {
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        additionalProperties[entry.key] = entry.value
                    }
                    return Context(additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                }
            }
        }
    }

    /**
     * Action properties
     */
    public data class Action(
        public val id: List<String>,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            val idArray = JsonArray(id.size)
            id.forEach { idArray.add(it) }
            json.add("id", idArray)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Action {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Action {
                try {
                    val id = jsonObject.get("id").asJsonArray.let { jsonArray ->
                        val collection = ArrayList<String>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(it.asString)
                        }
                        collection
                    }
                    return Action(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }
        }
    }

    /**
     * Resource properties
     * @param id UUID of the resource
     * @param type Resource type
     * @param method HTTP method of the resource
     * @param url URL of the resource
     * @param statusCode HTTP status code of the resource
     * @param duration Duration of the resource
     * @param size Size in octet of the resource response body
     * @param redirect Redirect phase properties
     * @param dns DNS phase properties
     * @param connect Connect phase properties
     * @param ssl SSL phase properties
     * @param firstByte First Byte phase properties
     * @param download Download phase properties
     * @param provider The provider for this resource
     */
    public data class Resource(
        public val id: String? = null,
        public val type: ResourceType,
        public val method: Method? = null,
        public var url: String,
        public val statusCode: Long? = null,
        public val duration: Long? = null,
        public val size: Long? = null,
        public val redirect: Redirect? = null,
        public val dns: Dns? = null,
        public val connect: Connect? = null,
        public val ssl: Ssl? = null,
        public val firstByte: FirstByte? = null,
        public val download: Download? = null,
        public val provider: Provider? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            json.add("type", type.toJson())
            method?.let { methodNonNull ->
                json.add("method", methodNonNull.toJson())
            }
            json.addProperty("url", url)
            statusCode?.let { statusCodeNonNull ->
                json.addProperty("status_code", statusCodeNonNull)
            }
            duration?.let { durationNonNull ->
                json.addProperty("duration", durationNonNull)
            }
            size?.let { sizeNonNull ->
                json.addProperty("size", sizeNonNull)
            }
            redirect?.let { redirectNonNull ->
                json.add("redirect", redirectNonNull.toJson())
            }
            dns?.let { dnsNonNull ->
                json.add("dns", dnsNonNull.toJson())
            }
            connect?.let { connectNonNull ->
                json.add("connect", connectNonNull.toJson())
            }
            ssl?.let { sslNonNull ->
                json.add("ssl", sslNonNull.toJson())
            }
            firstByte?.let { firstByteNonNull ->
                json.add("first_byte", firstByteNonNull.toJson())
            }
            download?.let { downloadNonNull ->
                json.add("download", downloadNonNull.toJson())
            }
            provider?.let { providerNonNull ->
                json.add("provider", providerNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Resource {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Resource {
                try {
                    val id = jsonObject.get("id")?.asString
                    val type = ResourceType.fromJson(jsonObject.get("type").asString)
                    val method = jsonObject.get("method")?.asString?.let {
                        Method.fromJson(it)
                    }
                    val url = jsonObject.get("url").asString
                    val statusCode = jsonObject.get("status_code")?.asLong
                    val duration = jsonObject.get("duration")?.asLong
                    val size = jsonObject.get("size")?.asLong
                    val redirect = jsonObject.get("redirect")?.asJsonObject?.let {
                        Redirect.fromJsonObject(it)
                    }
                    val dns = jsonObject.get("dns")?.asJsonObject?.let {
                        Dns.fromJsonObject(it)
                    }
                    val connect = jsonObject.get("connect")?.asJsonObject?.let {
                        Connect.fromJsonObject(it)
                    }
                    val ssl = jsonObject.get("ssl")?.asJsonObject?.let {
                        Ssl.fromJsonObject(it)
                    }
                    val firstByte = jsonObject.get("first_byte")?.asJsonObject?.let {
                        FirstByte.fromJsonObject(it)
                    }
                    val download = jsonObject.get("download")?.asJsonObject?.let {
                        Download.fromJsonObject(it)
                    }
                    val provider = jsonObject.get("provider")?.asJsonObject?.let {
                        Provider.fromJsonObject(it)
                    }
                    return Resource(id, type, method, url, statusCode, duration, size, redirect,
                            dns, connect, ssl, firstByte, download, provider)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                }
            }
        }
    }

    /**
     * Cellular connectivity properties
     * @param technology The type of a radio technology used for cellular connection
     * @param carrierName The name of the SIM carrier
     */
    public data class Cellular(
        public val technology: String? = null,
        public val carrierName: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            technology?.let { technologyNonNull ->
                json.addProperty("technology", technologyNonNull)
            }
            carrierName?.let { carrierNameNonNull ->
                json.addProperty("carrier_name", carrierNameNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Cellular {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Cellular {
                try {
                    val technology = jsonObject.get("technology")?.asString
                    val carrierName = jsonObject.get("carrier_name")?.asString
                    return Cellular(technology, carrierName)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                }
            }
        }
    }

    /**
     * The viewport represents the rectangular area that is currently being viewed. Content outside
     * the viewport is not visible onscreen until scrolled into view.
     * @param width Width of the viewport (in pixels)
     * @param height Height of the viewport (in pixels)
     */
    public data class Viewport(
        public val width: Number,
        public val height: Number,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("width", width)
            json.addProperty("height", height)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Viewport {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Viewport {
                try {
                    val width = jsonObject.get("width").asNumber
                    val height = jsonObject.get("height").asNumber
                    return Viewport(width, height)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                }
            }
        }
    }

    /**
     * Session-related internal properties
     * @param plan Session plan: 1 is the plan without replay, 2 is the plan with replay
     * (deprecated)
     */
    public data class DdSession(
        public val plan: Plan? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            plan?.let { planNonNull ->
                json.add("plan", planNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): DdSession {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): DdSession {
                try {
                    val plan = jsonObject.get("plan")?.asString?.let {
                        Plan.fromJson(it)
                    }
                    return DdSession(plan)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                }
            }
        }
    }

    /**
     * Redirect phase properties
     * @param duration Duration in ns of the resource redirect phase
     * @param start Duration in ns between start of the request and start of the redirect phase
     */
    public data class Redirect(
        public val duration: Long,
        public val start: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("duration", duration)
            json.addProperty("start", start)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Redirect {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Redirect",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Redirect {
                try {
                    val duration = jsonObject.get("duration").asLong
                    val start = jsonObject.get("start").asLong
                    return Redirect(duration, start)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Redirect",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Redirect",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Redirect",
                        e
                    )
                }
            }
        }
    }

    /**
     * DNS phase properties
     * @param duration Duration in ns of the resource dns phase
     * @param start Duration in ns between start of the request and start of the dns phase
     */
    public data class Dns(
        public val duration: Long,
        public val start: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("duration", duration)
            json.addProperty("start", start)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Dns {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dns",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Dns {
                try {
                    val duration = jsonObject.get("duration").asLong
                    val start = jsonObject.get("start").asLong
                    return Dns(duration, start)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dns",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dns",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dns",
                        e
                    )
                }
            }
        }
    }

    /**
     * Connect phase properties
     * @param duration Duration in ns of the resource connect phase
     * @param start Duration in ns between start of the request and start of the connect phase
     */
    public data class Connect(
        public val duration: Long,
        public val start: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("duration", duration)
            json.addProperty("start", start)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Connect {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connect",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Connect {
                try {
                    val duration = jsonObject.get("duration").asLong
                    val start = jsonObject.get("start").asLong
                    return Connect(duration, start)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connect",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connect",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connect",
                        e
                    )
                }
            }
        }
    }

    /**
     * SSL phase properties
     * @param duration Duration in ns of the resource ssl phase
     * @param start Duration in ns between start of the request and start of the ssl phase
     */
    public data class Ssl(
        public val duration: Long,
        public val start: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("duration", duration)
            json.addProperty("start", start)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Ssl {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Ssl",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Ssl {
                try {
                    val duration = jsonObject.get("duration").asLong
                    val start = jsonObject.get("start").asLong
                    return Ssl(duration, start)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Ssl",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Ssl",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Ssl",
                        e
                    )
                }
            }
        }
    }

    /**
     * First Byte phase properties
     * @param duration Duration in ns of the resource first byte phase
     * @param start Duration in ns between start of the request and start of the first byte phase
     */
    public data class FirstByte(
        public val duration: Long,
        public val start: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("duration", duration)
            json.addProperty("start", start)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): FirstByte {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type FirstByte",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): FirstByte {
                try {
                    val duration = jsonObject.get("duration").asLong
                    val start = jsonObject.get("start").asLong
                    return FirstByte(duration, start)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type FirstByte",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type FirstByte",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type FirstByte",
                        e
                    )
                }
            }
        }
    }

    /**
     * Download phase properties
     * @param duration Duration in ns of the resource download phase
     * @param start Duration in ns between start of the request and start of the download phase
     */
    public data class Download(
        public val duration: Long,
        public val start: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("duration", duration)
            json.addProperty("start", start)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Download {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Download",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Download {
                try {
                    val duration = jsonObject.get("duration").asLong
                    val start = jsonObject.get("start").asLong
                    return Download(duration, start)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Download",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Download",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Download",
                        e
                    )
                }
            }
        }
    }

    /**
     * The provider for this resource
     * @param domain The domain name of the provider
     * @param name The user friendly name of the provider
     * @param type The type of provider
     */
    public data class Provider(
        public val domain: String? = null,
        public val name: String? = null,
        public val type: ProviderType? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            domain?.let { domainNonNull ->
                json.addProperty("domain", domainNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            type?.let { typeNonNull ->
                json.add("type", typeNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Provider {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Provider",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Provider {
                try {
                    val domain = jsonObject.get("domain")?.asString
                    val name = jsonObject.get("name")?.asString
                    val type = jsonObject.get("type")?.asString?.let {
                        ProviderType.fromJson(it)
                    }
                    return Provider(domain, name, type)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Provider",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Provider",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Provider",
                        e
                    )
                }
            }
        }
    }

    /**
     * The source of this event
     */
    public enum class Source(
        private val jsonValue: String,
    ) {
        ANDROID("android"),
        IOS("ios"),
        BROWSER("browser"),
        FLUTTER("flutter"),
        REACT_NATIVE("react-native"),
        ROKU("roku"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Source = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Type of the session
     */
    public enum class ResourceEventSessionType(
        private val jsonValue: String,
    ) {
        USER("user"),
        SYNTHETICS("synthetics"),
        CI_TEST("ci_test"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): ResourceEventSessionType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Status of the device connectivity
     */
    public enum class Status(
        private val jsonValue: String,
    ) {
        CONNECTED("connected"),
        NOT_CONNECTED("not_connected"),
        MAYBE("maybe"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Status = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    public enum class Interface(
        private val jsonValue: String,
    ) {
        BLUETOOTH("bluetooth"),
        CELLULAR("cellular"),
        ETHERNET("ethernet"),
        WIFI("wifi"),
        WIMAX("wimax"),
        MIXED("mixed"),
        OTHER("other"),
        UNKNOWN("unknown"),
        NONE("none"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Interface = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Device type info
     */
    public enum class DeviceType(
        private val jsonValue: String,
    ) {
        MOBILE("mobile"),
        DESKTOP("desktop"),
        TABLET("tablet"),
        TV("tv"),
        GAMING_CONSOLE("gaming_console"),
        BOT("bot"),
        OTHER("other"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): DeviceType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Resource type
     */
    public enum class ResourceType(
        private val jsonValue: String,
    ) {
        DOCUMENT("document"),
        XHR("xhr"),
        BEACON("beacon"),
        FETCH("fetch"),
        CSS("css"),
        JS("js"),
        IMAGE("image"),
        FONT("font"),
        MEDIA("media"),
        OTHER("other"),
        NATIVE("native"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): ResourceType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * HTTP method of the resource
     */
    public enum class Method(
        private val jsonValue: String,
    ) {
        POST("POST"),
        GET("GET"),
        HEAD("HEAD"),
        PUT("PUT"),
        DELETE("DELETE"),
        PATCH("PATCH"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Method = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Session plan: 1 is the plan without replay, 2 is the plan with replay (deprecated)
     */
    public enum class Plan(
        private val jsonValue: Number,
    ) {
        PLAN_1(1),
        PLAN_2(2),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Plan = values().first {
                it.jsonValue.toString() == jsonString
            }
        }
    }

    /**
     * The type of provider
     */
    public enum class ProviderType(
        private val jsonValue: String,
    ) {
        AD("ad"),
        ADVERTISING("advertising"),
        ANALYTICS("analytics"),
        CDN("cdn"),
        CONTENT("content"),
        CUSTOMER_SUCCESS("customer-success"),
        FIRST_PARTY("first party"),
        HOSTING("hosting"),
        MARKETING("marketing"),
        OTHER("other"),
        SOCIAL("social"),
        TAG_MANAGER("tag-manager"),
        UTILITY("utility"),
        VIDEO("video"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): ProviderType = values().first {
                it.jsonValue == jsonString
            }
        }
    }
}
