package com.datadog.android.rum.model

import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Number
import kotlin.String
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Schema of properties for a technical performance metric
 * @param min The minimum value seen for this metric during the view's lifetime.
 * @param max The maximum value seen for this metric during the view's lifetime.
 * @param average The average value for this metric during the view's lifetime.
 * @param metricMax The maximum possible value we could see for this metric, if such a max is
 * relevant and can vary from session to session.
 */
public data class RumPerfMetric(
    public val min: Number,
    public val max: Number,
    public val average: Number,
    public val metricMax: Number? = null,
) {
    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.addProperty("min", min)
        json.addProperty("max", max)
        json.addProperty("average", average)
        metricMax?.let { metricMaxNonNull ->
            json.addProperty("metric_max", metricMaxNonNull)
        }
        return json
    }

    public companion object {
        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): RumPerfMetric {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type RumPerfMetric",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): RumPerfMetric {
            try {
                val min = jsonObject.get("min").asNumber
                val max = jsonObject.get("max").asNumber
                val average = jsonObject.get("average").asNumber
                val metricMax = jsonObject.get("metric_max")?.asNumber
                return RumPerfMetric(min, max, average, metricMax)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type RumPerfMetric",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type RumPerfMetric",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type RumPerfMetric",
                    e
                )
            }
        }
    }
}
