package com.datadog.android.rum.model

import com.datadog.android.core.`internal`.utils.JsonSerializer
import com.google.gson.JsonArray
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import com.google.gson.JsonPrimitive
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Any
import kotlin.Array
import kotlin.Boolean
import kotlin.Long
import kotlin.Number
import kotlin.String
import kotlin.collections.ArrayList
import kotlin.collections.List
import kotlin.collections.MutableMap
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Schema of common properties of RUM events
 * @param date Start of the event in ms from epoch
 * @param application Application properties
 * @param service The service name for this application
 * @param version The version for this application
 * @param session Session properties
 * Session properties
 * @param source The source of this event
 * @param view View properties
 * View properties
 * @param usr User properties
 * @param connectivity Device connectivity properties
 * @param display Display properties
 * @param synthetics Synthetics properties
 * @param ciTest CI Visibility properties
 * @param os Operating system properties
 * @param device Device properties
 * @param dd Internal properties
 * Internal properties
 * @param context User provided context
 * @param featureFlags Feature flags properties
 */
public data class ViewEvent(
    public val date: Long,
    public val application: Application,
    public val service: String? = null,
    public val version: String? = null,
    public val session: ViewEventSession,
    public val source: Source? = null,
    public val view: View,
    public val usr: Usr? = null,
    public val connectivity: Connectivity? = null,
    public val display: Display? = null,
    public val synthetics: Synthetics? = null,
    public val ciTest: CiTest? = null,
    public val os: Os? = null,
    public val device: Device? = null,
    public val dd: Dd,
    public val context: Context? = null,
    public val featureFlags: Context? = null,
) {
    public val type: String = "view"

    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.addProperty("date", date)
        json.add("application", application.toJson())
        service?.let { serviceNonNull ->
            json.addProperty("service", serviceNonNull)
        }
        version?.let { versionNonNull ->
            json.addProperty("version", versionNonNull)
        }
        json.add("session", session.toJson())
        source?.let { sourceNonNull ->
            json.add("source", sourceNonNull.toJson())
        }
        json.add("view", view.toJson())
        usr?.let { usrNonNull ->
            json.add("usr", usrNonNull.toJson())
        }
        connectivity?.let { connectivityNonNull ->
            json.add("connectivity", connectivityNonNull.toJson())
        }
        display?.let { displayNonNull ->
            json.add("display", displayNonNull.toJson())
        }
        synthetics?.let { syntheticsNonNull ->
            json.add("synthetics", syntheticsNonNull.toJson())
        }
        ciTest?.let { ciTestNonNull ->
            json.add("ci_test", ciTestNonNull.toJson())
        }
        os?.let { osNonNull ->
            json.add("os", osNonNull.toJson())
        }
        device?.let { deviceNonNull ->
            json.add("device", deviceNonNull.toJson())
        }
        json.add("_dd", dd.toJson())
        context?.let { contextNonNull ->
            json.add("context", contextNonNull.toJson())
        }
        json.addProperty("type", type)
        featureFlags?.let { featureFlagsNonNull ->
            json.add("feature_flags", featureFlagsNonNull.toJson())
        }
        return json
    }

    public companion object {
        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): ViewEvent {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type ViewEvent",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): ViewEvent {
            try {
                val date = jsonObject.get("date").asLong
                val application = jsonObject.get("application").asJsonObject.let {
                    Application.fromJsonObject(it)
                }
                val service = jsonObject.get("service")?.asString
                val version = jsonObject.get("version")?.asString
                val session = jsonObject.get("session").asJsonObject.let {
                    ViewEventSession.fromJsonObject(it)
                }
                val source = jsonObject.get("source")?.asString?.let {
                    Source.fromJson(it)
                }
                val view = jsonObject.get("view").asJsonObject.let {
                    View.fromJsonObject(it)
                }
                val usr = jsonObject.get("usr")?.asJsonObject?.let {
                    Usr.fromJsonObject(it)
                }
                val connectivity = jsonObject.get("connectivity")?.asJsonObject?.let {
                    Connectivity.fromJsonObject(it)
                }
                val display = jsonObject.get("display")?.asJsonObject?.let {
                    Display.fromJsonObject(it)
                }
                val synthetics = jsonObject.get("synthetics")?.asJsonObject?.let {
                    Synthetics.fromJsonObject(it)
                }
                val ciTest = jsonObject.get("ci_test")?.asJsonObject?.let {
                    CiTest.fromJsonObject(it)
                }
                val os = jsonObject.get("os")?.asJsonObject?.let {
                    Os.fromJsonObject(it)
                }
                val device = jsonObject.get("device")?.asJsonObject?.let {
                    Device.fromJsonObject(it)
                }
                val dd = jsonObject.get("_dd").asJsonObject.let {
                    Dd.fromJsonObject(it)
                }
                val context = jsonObject.get("context")?.asJsonObject?.let {
                    Context.fromJsonObject(it)
                }
                val featureFlags = jsonObject.get("feature_flags")?.asJsonObject?.let {
                    Context.fromJsonObject(it)
                }
                return ViewEvent(date, application, service, version, session, source, view, usr,
                        connectivity, display, synthetics, ciTest, os, device, dd, context,
                        featureFlags)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type ViewEvent",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type ViewEvent",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type ViewEvent",
                    e
                )
            }
        }
    }

    /**
     * Application properties
     * @param id UUID of the application
     */
    public data class Application(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Application {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Application {
                try {
                    val id = jsonObject.get("id").asString
                    return Application(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }
        }
    }

    /**
     * Session properties
     * Session properties
     * @param id UUID of the session
     * @param type Type of the session
     * @param hasReplay Whether this session has a replay
     * @param startPrecondition The precondition that led to the creation of the session
     * @param isActive Whether this session is currently active. Set to false to manually stop a
     * session
     * @param sampledForReplay Whether this session has been sampled for replay
     */
    public data class ViewEventSession(
        public val id: String,
        public val type: ViewEventSessionType,
        public val hasReplay: Boolean? = null,
        public val startPrecondition: StartPrecondition? = null,
        public val isActive: Boolean? = true,
        public val sampledForReplay: Boolean? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            json.add("type", type.toJson())
            hasReplay?.let { hasReplayNonNull ->
                json.addProperty("has_replay", hasReplayNonNull)
            }
            startPrecondition?.let { startPreconditionNonNull ->
                json.add("start_precondition", startPreconditionNonNull.toJson())
            }
            isActive?.let { isActiveNonNull ->
                json.addProperty("is_active", isActiveNonNull)
            }
            sampledForReplay?.let { sampledForReplayNonNull ->
                json.addProperty("sampled_for_replay", sampledForReplayNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): ViewEventSession {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ViewEventSession",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): ViewEventSession {
                try {
                    val id = jsonObject.get("id").asString
                    val type = ViewEventSessionType.fromJson(jsonObject.get("type").asString)
                    val hasReplay = jsonObject.get("has_replay")?.asBoolean
                    val startPrecondition = jsonObject.get("start_precondition")?.asString?.let {
                        StartPrecondition.fromJson(it)
                    }
                    val isActive = jsonObject.get("is_active")?.asBoolean
                    val sampledForReplay = jsonObject.get("sampled_for_replay")?.asBoolean
                    return ViewEventSession(id, type, hasReplay, startPrecondition, isActive,
                            sampledForReplay)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ViewEventSession",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type ViewEventSession",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type ViewEventSession",
                        e
                    )
                }
            }
        }
    }

    /**
     * View properties
     * View properties
     * @param id UUID of the view
     * @param referrer URL that linked to the initial view of the page
     * @param url URL of the view
     * @param name User defined name of the view
     * @param loadingTime Duration in ns to the view is considered loaded
     * @param loadingType Type of the loading of the view
     * @param timeSpent Time spent on the view in ns
     * @param firstContentfulPaint Duration in ns to the first rendering
     * @param largestContentfulPaint Duration in ns to the largest contentful paint
     * @param firstInputDelay Duration in ns of the first input event delay
     * @param firstInputTime Duration in ns to the first input
     * @param cumulativeLayoutShift Total layout shift score that occurred on the view
     * @param domComplete Duration in ns to the complete parsing and loading of the document and its
     * sub resources
     * @param domContentLoaded Duration in ns to the complete parsing and loading of the document
     * without its sub resources
     * @param domInteractive Duration in ns to the end of the parsing of the document
     * @param loadEvent Duration in ns to the end of the load event handler execution
     * @param firstByte Duration in ns to the response start of the document request
     * @param customTimings User custom timings of the view. As timing name is used as facet path,
     * it must contain only letters, digits, or the characters - _ . @ $
     * @param isActive Whether the View corresponding to this event is considered active
     * @param isSlowRendered Whether the View had a low average refresh rate
     * @param action Properties of the actions of the view
     * @param error Properties of the errors of the view
     * @param crash Properties of the crashes of the view
     * @param longTask Properties of the long tasks of the view
     * @param frozenFrame Properties of the frozen frames of the view
     * @param resource Properties of the resources of the view
     * @param frustration Properties of the frustrations of the view
     * @param inForegroundPeriods List of the periods of time the user had the view in foreground
     * (focused in the browser)
     * @param memoryAverage Average memory used during the view lifetime (in bytes)
     * @param memoryMax Peak memory used during the view lifetime (in bytes)
     * @param cpuTicksCount Total number of cpu ticks during the view’s lifetime
     * @param cpuTicksPerSecond Average number of cpu ticks per second during the view’s lifetime
     * @param refreshRateAverage Average refresh rate during the view’s lifetime (in frames per
     * second)
     * @param refreshRateMin Minimum refresh rate during the view’s lifetime (in frames per second)
     * @param flutterBuildTime Schema of properties for a technical performance metric
     * @param flutterRasterTime Schema of properties for a technical performance metric
     * @param jsRefreshRate Schema of properties for a technical performance metric
     */
    public data class View(
        public val id: String,
        public var referrer: String? = null,
        public var url: String,
        public var name: String? = null,
        public val loadingTime: Long? = null,
        public val loadingType: LoadingType? = null,
        public val timeSpent: Long,
        public val firstContentfulPaint: Long? = null,
        public val largestContentfulPaint: Long? = null,
        public val firstInputDelay: Long? = null,
        public val firstInputTime: Long? = null,
        public val cumulativeLayoutShift: Number? = null,
        public val domComplete: Long? = null,
        public val domContentLoaded: Long? = null,
        public val domInteractive: Long? = null,
        public val loadEvent: Long? = null,
        public val firstByte: Long? = null,
        public val customTimings: CustomTimings? = null,
        public val isActive: Boolean? = null,
        public val isSlowRendered: Boolean? = null,
        public val action: Action,
        public val error: Error,
        public val crash: Crash? = null,
        public val longTask: LongTask? = null,
        public val frozenFrame: FrozenFrame? = null,
        public val resource: Resource,
        public val frustration: Frustration? = null,
        public val inForegroundPeriods: List<InForegroundPeriod>? = null,
        public val memoryAverage: Number? = null,
        public val memoryMax: Number? = null,
        public val cpuTicksCount: Number? = null,
        public val cpuTicksPerSecond: Number? = null,
        public val refreshRateAverage: Number? = null,
        public val refreshRateMin: Number? = null,
        public val flutterBuildTime: FlutterBuildTime? = null,
        public val flutterRasterTime: FlutterBuildTime? = null,
        public val jsRefreshRate: FlutterBuildTime? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            referrer?.let { referrerNonNull ->
                json.addProperty("referrer", referrerNonNull)
            }
            json.addProperty("url", url)
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            loadingTime?.let { loadingTimeNonNull ->
                json.addProperty("loading_time", loadingTimeNonNull)
            }
            loadingType?.let { loadingTypeNonNull ->
                json.add("loading_type", loadingTypeNonNull.toJson())
            }
            json.addProperty("time_spent", timeSpent)
            firstContentfulPaint?.let { firstContentfulPaintNonNull ->
                json.addProperty("first_contentful_paint", firstContentfulPaintNonNull)
            }
            largestContentfulPaint?.let { largestContentfulPaintNonNull ->
                json.addProperty("largest_contentful_paint", largestContentfulPaintNonNull)
            }
            firstInputDelay?.let { firstInputDelayNonNull ->
                json.addProperty("first_input_delay", firstInputDelayNonNull)
            }
            firstInputTime?.let { firstInputTimeNonNull ->
                json.addProperty("first_input_time", firstInputTimeNonNull)
            }
            cumulativeLayoutShift?.let { cumulativeLayoutShiftNonNull ->
                json.addProperty("cumulative_layout_shift", cumulativeLayoutShiftNonNull)
            }
            domComplete?.let { domCompleteNonNull ->
                json.addProperty("dom_complete", domCompleteNonNull)
            }
            domContentLoaded?.let { domContentLoadedNonNull ->
                json.addProperty("dom_content_loaded", domContentLoadedNonNull)
            }
            domInteractive?.let { domInteractiveNonNull ->
                json.addProperty("dom_interactive", domInteractiveNonNull)
            }
            loadEvent?.let { loadEventNonNull ->
                json.addProperty("load_event", loadEventNonNull)
            }
            firstByte?.let { firstByteNonNull ->
                json.addProperty("first_byte", firstByteNonNull)
            }
            customTimings?.let { customTimingsNonNull ->
                json.add("custom_timings", customTimingsNonNull.toJson())
            }
            isActive?.let { isActiveNonNull ->
                json.addProperty("is_active", isActiveNonNull)
            }
            isSlowRendered?.let { isSlowRenderedNonNull ->
                json.addProperty("is_slow_rendered", isSlowRenderedNonNull)
            }
            json.add("action", action.toJson())
            json.add("error", error.toJson())
            crash?.let { crashNonNull ->
                json.add("crash", crashNonNull.toJson())
            }
            longTask?.let { longTaskNonNull ->
                json.add("long_task", longTaskNonNull.toJson())
            }
            frozenFrame?.let { frozenFrameNonNull ->
                json.add("frozen_frame", frozenFrameNonNull.toJson())
            }
            json.add("resource", resource.toJson())
            frustration?.let { frustrationNonNull ->
                json.add("frustration", frustrationNonNull.toJson())
            }
            inForegroundPeriods?.let { inForegroundPeriodsNonNull ->
                val inForegroundPeriodsArray = JsonArray(inForegroundPeriodsNonNull.size)
                inForegroundPeriodsNonNull.forEach { inForegroundPeriodsArray.add(it.toJson()) }
                json.add("in_foreground_periods", inForegroundPeriodsArray)
            }
            memoryAverage?.let { memoryAverageNonNull ->
                json.addProperty("memory_average", memoryAverageNonNull)
            }
            memoryMax?.let { memoryMaxNonNull ->
                json.addProperty("memory_max", memoryMaxNonNull)
            }
            cpuTicksCount?.let { cpuTicksCountNonNull ->
                json.addProperty("cpu_ticks_count", cpuTicksCountNonNull)
            }
            cpuTicksPerSecond?.let { cpuTicksPerSecondNonNull ->
                json.addProperty("cpu_ticks_per_second", cpuTicksPerSecondNonNull)
            }
            refreshRateAverage?.let { refreshRateAverageNonNull ->
                json.addProperty("refresh_rate_average", refreshRateAverageNonNull)
            }
            refreshRateMin?.let { refreshRateMinNonNull ->
                json.addProperty("refresh_rate_min", refreshRateMinNonNull)
            }
            flutterBuildTime?.let { flutterBuildTimeNonNull ->
                json.add("flutter_build_time", flutterBuildTimeNonNull.toJson())
            }
            flutterRasterTime?.let { flutterRasterTimeNonNull ->
                json.add("flutter_raster_time", flutterRasterTimeNonNull.toJson())
            }
            jsRefreshRate?.let { jsRefreshRateNonNull ->
                json.add("js_refresh_rate", jsRefreshRateNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): View {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): View {
                try {
                    val id = jsonObject.get("id").asString
                    val referrer = jsonObject.get("referrer")?.asString
                    val url = jsonObject.get("url").asString
                    val name = jsonObject.get("name")?.asString
                    val loadingTime = jsonObject.get("loading_time")?.asLong
                    val loadingType = jsonObject.get("loading_type")?.asString?.let {
                        LoadingType.fromJson(it)
                    }
                    val timeSpent = jsonObject.get("time_spent").asLong
                    val firstContentfulPaint = jsonObject.get("first_contentful_paint")?.asLong
                    val largestContentfulPaint = jsonObject.get("largest_contentful_paint")?.asLong
                    val firstInputDelay = jsonObject.get("first_input_delay")?.asLong
                    val firstInputTime = jsonObject.get("first_input_time")?.asLong
                    val cumulativeLayoutShift = jsonObject.get("cumulative_layout_shift")?.asNumber
                    val domComplete = jsonObject.get("dom_complete")?.asLong
                    val domContentLoaded = jsonObject.get("dom_content_loaded")?.asLong
                    val domInteractive = jsonObject.get("dom_interactive")?.asLong
                    val loadEvent = jsonObject.get("load_event")?.asLong
                    val firstByte = jsonObject.get("first_byte")?.asLong
                    val customTimings = jsonObject.get("custom_timings")?.asJsonObject?.let {
                        CustomTimings.fromJsonObject(it)
                    }
                    val isActive = jsonObject.get("is_active")?.asBoolean
                    val isSlowRendered = jsonObject.get("is_slow_rendered")?.asBoolean
                    val action = jsonObject.get("action").asJsonObject.let {
                        Action.fromJsonObject(it)
                    }
                    val error = jsonObject.get("error").asJsonObject.let {
                        Error.fromJsonObject(it)
                    }
                    val crash = jsonObject.get("crash")?.asJsonObject?.let {
                        Crash.fromJsonObject(it)
                    }
                    val longTask = jsonObject.get("long_task")?.asJsonObject?.let {
                        LongTask.fromJsonObject(it)
                    }
                    val frozenFrame = jsonObject.get("frozen_frame")?.asJsonObject?.let {
                        FrozenFrame.fromJsonObject(it)
                    }
                    val resource = jsonObject.get("resource").asJsonObject.let {
                        Resource.fromJsonObject(it)
                    }
                    val frustration = jsonObject.get("frustration")?.asJsonObject?.let {
                        Frustration.fromJsonObject(it)
                    }
                    val inForegroundPeriods =
                            jsonObject.get("in_foreground_periods")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<InForegroundPeriod>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(InForegroundPeriod.fromJsonObject(it.asJsonObject))
                        }
                        collection
                    }
                    val memoryAverage = jsonObject.get("memory_average")?.asNumber
                    val memoryMax = jsonObject.get("memory_max")?.asNumber
                    val cpuTicksCount = jsonObject.get("cpu_ticks_count")?.asNumber
                    val cpuTicksPerSecond = jsonObject.get("cpu_ticks_per_second")?.asNumber
                    val refreshRateAverage = jsonObject.get("refresh_rate_average")?.asNumber
                    val refreshRateMin = jsonObject.get("refresh_rate_min")?.asNumber
                    val flutterBuildTime = jsonObject.get("flutter_build_time")?.asJsonObject?.let {
                        FlutterBuildTime.fromJsonObject(it)
                    }
                    val flutterRasterTime =
                            jsonObject.get("flutter_raster_time")?.asJsonObject?.let {
                        FlutterBuildTime.fromJsonObject(it)
                    }
                    val jsRefreshRate = jsonObject.get("js_refresh_rate")?.asJsonObject?.let {
                        FlutterBuildTime.fromJsonObject(it)
                    }
                    return View(id, referrer, url, name, loadingTime, loadingType, timeSpent,
                            firstContentfulPaint, largestContentfulPaint, firstInputDelay,
                            firstInputTime, cumulativeLayoutShift, domComplete, domContentLoaded,
                            domInteractive, loadEvent, firstByte, customTimings, isActive,
                            isSlowRendered, action, error, crash, longTask, frozenFrame, resource,
                            frustration, inForegroundPeriods, memoryAverage, memoryMax,
                            cpuTicksCount, cpuTicksPerSecond, refreshRateAverage, refreshRateMin,
                            flutterBuildTime, flutterRasterTime, jsRefreshRate)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }
        }
    }

    /**
     * User properties
     * @param id Identifier of the user
     * @param name Name of the user
     * @param email Email of the user
     */
    public data class Usr(
        public val id: String? = null,
        public val name: String? = null,
        public val email: String? = null,
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            email?.let { emailNonNull ->
                json.addProperty("email", emailNonNull)
            }
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.add(k, JsonSerializer.toJsonElement(v))
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("id", "name", "email")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Usr {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Usr {
                try {
                    val id = jsonObject.get("id")?.asString
                    val name = jsonObject.get("name")?.asString
                    val email = jsonObject.get("email")?.asString
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value
                        }
                    }
                    return Usr(id, name, email, additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                }
            }
        }
    }

    /**
     * Device connectivity properties
     * @param status Status of the device connectivity
     * @param interfaces The list of available network interfaces
     * @param cellular Cellular connectivity properties
     */
    public data class Connectivity(
        public val status: Status,
        public val interfaces: List<Interface>,
        public val cellular: Cellular? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("status", status.toJson())
            val interfacesArray = JsonArray(interfaces.size)
            interfaces.forEach { interfacesArray.add(it.toJson()) }
            json.add("interfaces", interfacesArray)
            cellular?.let { cellularNonNull ->
                json.add("cellular", cellularNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Connectivity {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Connectivity {
                try {
                    val status = Status.fromJson(jsonObject.get("status").asString)
                    val interfaces = jsonObject.get("interfaces").asJsonArray.let { jsonArray ->
                        val collection = ArrayList<Interface>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(Interface.fromJson(it.asString))
                        }
                        collection
                    }
                    val cellular = jsonObject.get("cellular")?.asJsonObject?.let {
                        Cellular.fromJsonObject(it)
                    }
                    return Connectivity(status, interfaces, cellular)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Connectivity",
                        e
                    )
                }
            }
        }
    }

    /**
     * Display properties
     * @param viewport The viewport represents the rectangular area that is currently being viewed.
     * Content outside the viewport is not visible onscreen until scrolled into view.
     */
    public data class Display(
        public val viewport: Viewport? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            viewport?.let { viewportNonNull ->
                json.add("viewport", viewportNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Display {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Display {
                try {
                    val viewport = jsonObject.get("viewport")?.asJsonObject?.let {
                        Viewport.fromJsonObject(it)
                    }
                    return Display(viewport)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Display",
                        e
                    )
                }
            }
        }
    }

    /**
     * Synthetics properties
     * @param testId The identifier of the current Synthetics test
     * @param resultId The identifier of the current Synthetics test results
     * @param injected Whether the event comes from a SDK instance injected by Synthetics
     */
    public data class Synthetics(
        public val testId: String,
        public val resultId: String,
        public val injected: Boolean? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("test_id", testId)
            json.addProperty("result_id", resultId)
            injected?.let { injectedNonNull ->
                json.addProperty("injected", injectedNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Synthetics {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Synthetics {
                try {
                    val testId = jsonObject.get("test_id").asString
                    val resultId = jsonObject.get("result_id").asString
                    val injected = jsonObject.get("injected")?.asBoolean
                    return Synthetics(testId, resultId, injected)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Synthetics",
                        e
                    )
                }
            }
        }
    }

    /**
     * CI Visibility properties
     * @param testExecutionId The identifier of the current CI Visibility test execution
     */
    public data class CiTest(
        public val testExecutionId: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("test_execution_id", testExecutionId)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): CiTest {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): CiTest {
                try {
                    val testExecutionId = jsonObject.get("test_execution_id").asString
                    return CiTest(testExecutionId)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type CiTest",
                        e
                    )
                }
            }
        }
    }

    /**
     * Operating system properties
     * @param name Operating system name, e.g. Android, iOS
     * @param version Full operating system version, e.g. 8.1.1
     * @param versionMajor Major operating system version, e.g. 8
     */
    public data class Os(
        public val name: String,
        public val version: String,
        public val versionMajor: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("name", name)
            json.addProperty("version", version)
            json.addProperty("version_major", versionMajor)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Os {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Os {
                try {
                    val name = jsonObject.get("name").asString
                    val version = jsonObject.get("version").asString
                    val versionMajor = jsonObject.get("version_major").asString
                    return Os(name, version, versionMajor)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }
        }
    }

    /**
     * Device properties
     * @param type Device type info
     * @param name Device marketing name, e.g. Xiaomi Redmi Note 8 Pro, Pixel 5, etc.
     * @param model Device SKU model, e.g. Samsung SM-988GN, etc. Quite often name and model can be
     * the same.
     * @param brand Device marketing brand, e.g. Apple, OPPO, Xiaomi, etc.
     * @param architecture The CPU architecture of the device that is reporting the error
     */
    public data class Device(
        public val type: DeviceType,
        public val name: String? = null,
        public val model: String? = null,
        public val brand: String? = null,
        public val architecture: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("type", type.toJson())
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            model?.let { modelNonNull ->
                json.addProperty("model", modelNonNull)
            }
            brand?.let { brandNonNull ->
                json.addProperty("brand", brandNonNull)
            }
            architecture?.let { architectureNonNull ->
                json.addProperty("architecture", architectureNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Device {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Device {
                try {
                    val type = DeviceType.fromJson(jsonObject.get("type").asString)
                    val name = jsonObject.get("name")?.asString
                    val model = jsonObject.get("model")?.asString
                    val brand = jsonObject.get("brand")?.asString
                    val architecture = jsonObject.get("architecture")?.asString
                    return Device(type, name, model, brand, architecture)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }
        }
    }

    /**
     * Internal properties
     * Internal properties
     * @param session Session-related internal properties
     * @param browserSdkVersion Browser SDK version
     * @param documentVersion Version of the update of the view event
     * @param pageStates List of the page states during the view
     * @param replayStats Debug metadata for Replay Sessions
     */
    public data class Dd(
        public val session: DdSession? = null,
        public val browserSdkVersion: String? = null,
        public val documentVersion: Long,
        public val pageStates: List<PageState>? = null,
        public val replayStats: ReplayStats? = null,
    ) {
        public val formatVersion: Long = 2L

        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("format_version", formatVersion)
            session?.let { sessionNonNull ->
                json.add("session", sessionNonNull.toJson())
            }
            browserSdkVersion?.let { browserSdkVersionNonNull ->
                json.addProperty("browser_sdk_version", browserSdkVersionNonNull)
            }
            json.addProperty("document_version", documentVersion)
            pageStates?.let { pageStatesNonNull ->
                val pageStatesArray = JsonArray(pageStatesNonNull.size)
                pageStatesNonNull.forEach { pageStatesArray.add(it.toJson()) }
                json.add("page_states", pageStatesArray)
            }
            replayStats?.let { replayStatsNonNull ->
                json.add("replay_stats", replayStatsNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Dd {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Dd {
                try {
                    val session = jsonObject.get("session")?.asJsonObject?.let {
                        DdSession.fromJsonObject(it)
                    }
                    val browserSdkVersion = jsonObject.get("browser_sdk_version")?.asString
                    val documentVersion = jsonObject.get("document_version").asLong
                    val pageStates = jsonObject.get("page_states")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<PageState>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(PageState.fromJsonObject(it.asJsonObject))
                        }
                        collection
                    }
                    val replayStats = jsonObject.get("replay_stats")?.asJsonObject?.let {
                        ReplayStats.fromJsonObject(it)
                    }
                    return Dd(session, browserSdkVersion, documentVersion, pageStates, replayStats)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }
        }
    }

    /**
     * User provided context
     */
    public data class Context(
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            additionalProperties.forEach { (k, v) ->
                json.add(k, JsonSerializer.toJsonElement(v))
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Context {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Context {
                try {
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        additionalProperties[entry.key] = entry.value
                    }
                    return Context(additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Context",
                        e
                    )
                }
            }
        }
    }

    /**
     * User custom timings of the view. As timing name is used as facet path, it must contain only
     * letters, digits, or the characters - _ . @ $
     */
    public data class CustomTimings(
        public val additionalProperties: MutableMap<String, Long> = mutableMapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            additionalProperties.forEach { (k, v) ->
                json.addProperty(k, v)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): CustomTimings {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type CustomTimings",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): CustomTimings {
                try {
                    val additionalProperties = mutableMapOf<String, Long>()
                    for (entry in jsonObject.entrySet()) {
                        additionalProperties[entry.key] = entry.value.asLong
                    }
                    return CustomTimings(additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type CustomTimings",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type CustomTimings",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type CustomTimings",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the actions of the view
     * @param count Number of actions that occurred on the view
     */
    public data class Action(
        public val count: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("count", count)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Action {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Action {
                try {
                    val count = jsonObject.get("count").asLong
                    return Action(count)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the errors of the view
     * @param count Number of errors that occurred on the view
     */
    public data class Error(
        public val count: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("count", count)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Error {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Error {
                try {
                    val count = jsonObject.get("count").asLong
                    return Error(count)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the crashes of the view
     * @param count Number of crashes that occurred on the view
     */
    public data class Crash(
        public val count: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("count", count)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Crash {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Crash",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Crash {
                try {
                    val count = jsonObject.get("count").asLong
                    return Crash(count)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Crash",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Crash",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Crash",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the long tasks of the view
     * @param count Number of long tasks that occurred on the view
     */
    public data class LongTask(
        public val count: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("count", count)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): LongTask {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type LongTask",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): LongTask {
                try {
                    val count = jsonObject.get("count").asLong
                    return LongTask(count)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type LongTask",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type LongTask",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type LongTask",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the frozen frames of the view
     * @param count Number of frozen frames that occurred on the view
     */
    public data class FrozenFrame(
        public val count: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("count", count)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): FrozenFrame {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type FrozenFrame",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): FrozenFrame {
                try {
                    val count = jsonObject.get("count").asLong
                    return FrozenFrame(count)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type FrozenFrame",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type FrozenFrame",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type FrozenFrame",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the resources of the view
     * @param count Number of resources that occurred on the view
     */
    public data class Resource(
        public val count: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("count", count)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Resource {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Resource {
                try {
                    val count = jsonObject.get("count").asLong
                    return Resource(count)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Resource",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the frustrations of the view
     * @param count Number of frustrations that occurred on the view
     */
    public data class Frustration(
        public val count: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("count", count)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Frustration {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Frustration",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Frustration {
                try {
                    val count = jsonObject.get("count").asLong
                    return Frustration(count)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Frustration",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Frustration",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Frustration",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the foreground period of the view
     * @param start Duration in ns between start of the view and start of foreground period
     * @param duration Duration in ns of the view foreground period
     */
    public data class InForegroundPeriod(
        public val start: Long,
        public val duration: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("start", start)
            json.addProperty("duration", duration)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): InForegroundPeriod {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type InForegroundPeriod",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): InForegroundPeriod {
                try {
                    val start = jsonObject.get("start").asLong
                    val duration = jsonObject.get("duration").asLong
                    return InForegroundPeriod(start, duration)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type InForegroundPeriod",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type InForegroundPeriod",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type InForegroundPeriod",
                        e
                    )
                }
            }
        }
    }

    /**
     * Schema of properties for a technical performance metric
     * @param min The minimum value seen for this metric during the view's lifetime.
     * @param max The maximum value seen for this metric during the view's lifetime.
     * @param average The average value for this metric during the view's lifetime.
     * @param metricMax The maximum possible value we could see for this metric, if such a max is
     * relevant and can vary from session to session.
     */
    public data class FlutterBuildTime(
        public val min: Number,
        public val max: Number,
        public val average: Number,
        public val metricMax: Number? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("min", min)
            json.addProperty("max", max)
            json.addProperty("average", average)
            metricMax?.let { metricMaxNonNull ->
                json.addProperty("metric_max", metricMaxNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): FlutterBuildTime {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type FlutterBuildTime",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): FlutterBuildTime {
                try {
                    val min = jsonObject.get("min").asNumber
                    val max = jsonObject.get("max").asNumber
                    val average = jsonObject.get("average").asNumber
                    val metricMax = jsonObject.get("metric_max")?.asNumber
                    return FlutterBuildTime(min, max, average, metricMax)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type FlutterBuildTime",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type FlutterBuildTime",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type FlutterBuildTime",
                        e
                    )
                }
            }
        }
    }

    /**
     * Cellular connectivity properties
     * @param technology The type of a radio technology used for cellular connection
     * @param carrierName The name of the SIM carrier
     */
    public data class Cellular(
        public val technology: String? = null,
        public val carrierName: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            technology?.let { technologyNonNull ->
                json.addProperty("technology", technologyNonNull)
            }
            carrierName?.let { carrierNameNonNull ->
                json.addProperty("carrier_name", carrierNameNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Cellular {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Cellular {
                try {
                    val technology = jsonObject.get("technology")?.asString
                    val carrierName = jsonObject.get("carrier_name")?.asString
                    return Cellular(technology, carrierName)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Cellular",
                        e
                    )
                }
            }
        }
    }

    /**
     * The viewport represents the rectangular area that is currently being viewed. Content outside
     * the viewport is not visible onscreen until scrolled into view.
     * @param width Width of the viewport (in pixels)
     * @param height Height of the viewport (in pixels)
     */
    public data class Viewport(
        public val width: Number,
        public val height: Number,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("width", width)
            json.addProperty("height", height)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Viewport {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Viewport {
                try {
                    val width = jsonObject.get("width").asNumber
                    val height = jsonObject.get("height").asNumber
                    return Viewport(width, height)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Viewport",
                        e
                    )
                }
            }
        }
    }

    /**
     * Session-related internal properties
     * @param plan Session plan: 1 is the plan without replay, 2 is the plan with replay
     * (deprecated)
     */
    public data class DdSession(
        public val plan: Plan? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            plan?.let { planNonNull ->
                json.add("plan", planNonNull.toJson())
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): DdSession {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): DdSession {
                try {
                    val plan = jsonObject.get("plan")?.asString?.let {
                        Plan.fromJson(it)
                    }
                    return DdSession(plan)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type DdSession",
                        e
                    )
                }
            }
        }
    }

    /**
     * Properties of the page state
     * @param state Page state name
     * @param start Duration in ns between start of the view and start of the page state
     */
    public data class PageState(
        public val state: State,
        public val start: Long,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("state", state.toJson())
            json.addProperty("start", start)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): PageState {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type PageState",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): PageState {
                try {
                    val state = State.fromJson(jsonObject.get("state").asString)
                    val start = jsonObject.get("start").asLong
                    return PageState(state, start)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type PageState",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type PageState",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type PageState",
                        e
                    )
                }
            }
        }
    }

    /**
     * Debug metadata for Replay Sessions
     * @param recordsCount The number of records produced during this view lifetime
     * @param segmentsCount The number of segments sent during this view lifetime
     * @param segmentsTotalRawSize The total size in bytes of the segments sent during this view
     * lifetime
     */
    public data class ReplayStats(
        public val recordsCount: Long? = 0L,
        public val segmentsCount: Long? = 0L,
        public val segmentsTotalRawSize: Long? = 0L,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            recordsCount?.let { recordsCountNonNull ->
                json.addProperty("records_count", recordsCountNonNull)
            }
            segmentsCount?.let { segmentsCountNonNull ->
                json.addProperty("segments_count", segmentsCountNonNull)
            }
            segmentsTotalRawSize?.let { segmentsTotalRawSizeNonNull ->
                json.addProperty("segments_total_raw_size", segmentsTotalRawSizeNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): ReplayStats {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ReplayStats",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): ReplayStats {
                try {
                    val recordsCount = jsonObject.get("records_count")?.asLong
                    val segmentsCount = jsonObject.get("segments_count")?.asLong
                    val segmentsTotalRawSize = jsonObject.get("segments_total_raw_size")?.asLong
                    return ReplayStats(recordsCount, segmentsCount, segmentsTotalRawSize)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type ReplayStats",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type ReplayStats",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type ReplayStats",
                        e
                    )
                }
            }
        }
    }

    /**
     * The source of this event
     */
    public enum class Source(
        private val jsonValue: String,
    ) {
        ANDROID("android"),
        IOS("ios"),
        BROWSER("browser"),
        FLUTTER("flutter"),
        REACT_NATIVE("react-native"),
        ROKU("roku"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Source = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Type of the session
     */
    public enum class ViewEventSessionType(
        private val jsonValue: String,
    ) {
        USER("user"),
        SYNTHETICS("synthetics"),
        CI_TEST("ci_test"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): ViewEventSessionType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * The precondition that led to the creation of the session
     */
    public enum class StartPrecondition(
        private val jsonValue: String,
    ) {
        APP_LAUNCH("app_launch"),
        INACTIVITY_TIMEOUT("inactivity_timeout"),
        MAX_DURATION("max_duration"),
        EXPLICIT_STOP("explicit_stop"),
        BACKGROUND_EVENT("background_event"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): StartPrecondition = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Type of the loading of the view
     */
    public enum class LoadingType(
        private val jsonValue: String,
    ) {
        INITIAL_LOAD("initial_load"),
        ROUTE_CHANGE("route_change"),
        ACTIVITY_DISPLAY("activity_display"),
        ACTIVITY_REDISPLAY("activity_redisplay"),
        FRAGMENT_DISPLAY("fragment_display"),
        FRAGMENT_REDISPLAY("fragment_redisplay"),
        VIEW_CONTROLLER_DISPLAY("view_controller_display"),
        VIEW_CONTROLLER_REDISPLAY("view_controller_redisplay"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): LoadingType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Status of the device connectivity
     */
    public enum class Status(
        private val jsonValue: String,
    ) {
        CONNECTED("connected"),
        NOT_CONNECTED("not_connected"),
        MAYBE("maybe"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Status = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    public enum class Interface(
        private val jsonValue: String,
    ) {
        BLUETOOTH("bluetooth"),
        CELLULAR("cellular"),
        ETHERNET("ethernet"),
        WIFI("wifi"),
        WIMAX("wimax"),
        MIXED("mixed"),
        OTHER("other"),
        UNKNOWN("unknown"),
        NONE("none"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Interface = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Device type info
     */
    public enum class DeviceType(
        private val jsonValue: String,
    ) {
        MOBILE("mobile"),
        DESKTOP("desktop"),
        TABLET("tablet"),
        TV("tv"),
        GAMING_CONSOLE("gaming_console"),
        BOT("bot"),
        OTHER("other"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): DeviceType = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * Session plan: 1 is the plan without replay, 2 is the plan with replay (deprecated)
     */
    public enum class Plan(
        private val jsonValue: Number,
    ) {
        PLAN_1(1),
        PLAN_2(2),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Plan = values().first {
                it.jsonValue.toString() == jsonString
            }
        }
    }

    /**
     * Page state name
     */
    public enum class State(
        private val jsonValue: String,
    ) {
        ACTIVE("active"),
        PASSIVE("passive"),
        HIDDEN("hidden"),
        FROZEN("frozen"),
        TERMINATED("terminated"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): State = values().first {
                it.jsonValue == jsonString
            }
        }
    }
}
