package com.datadog.android.telemetry.model

import com.google.gson.JsonArray
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import com.google.gson.JsonPrimitive
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Boolean
import kotlin.Long
import kotlin.String
import kotlin.collections.ArrayList
import kotlin.collections.List
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Schema of common properties of Telemetry events
 * @param dd Internal properties
 * @param date Start of the event in ms from epoch
 * @param service The SDK generating the telemetry event
 * @param source The source of this event
 * @param version The version of the SDK generating the telemetry event
 * @param application Application properties
 * @param session Session properties
 * @param view View properties
 * @param action Action properties
 * @param experimentalFeatures Enabled experimental features
 * @param telemetry The telemetry configuration information
 */
public data class TelemetryConfigurationEvent(
    public val dd: Dd,
    public val date: Long,
    public val service: String,
    public val source: Source,
    public val version: String,
    public val application: Application? = null,
    public val session: Session? = null,
    public val view: View? = null,
    public val action: Action? = null,
    public val experimentalFeatures: List<String>? = null,
    public val telemetry: Telemetry,
) {
    public val type: String = "telemetry"

    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.add("_dd", dd.toJson())
        json.addProperty("type", type)
        json.addProperty("date", date)
        json.addProperty("service", service)
        json.add("source", source.toJson())
        json.addProperty("version", version)
        application?.let { applicationNonNull ->
            json.add("application", applicationNonNull.toJson())
        }
        session?.let { sessionNonNull ->
            json.add("session", sessionNonNull.toJson())
        }
        view?.let { viewNonNull ->
            json.add("view", viewNonNull.toJson())
        }
        action?.let { actionNonNull ->
            json.add("action", actionNonNull.toJson())
        }
        experimentalFeatures?.let { experimentalFeaturesNonNull ->
            val experimentalFeaturesArray = JsonArray(experimentalFeaturesNonNull.size)
            experimentalFeaturesNonNull.forEach { experimentalFeaturesArray.add(it) }
            json.add("experimental_features", experimentalFeaturesArray)
        }
        json.add("telemetry", telemetry.toJson())
        return json
    }

    public companion object {
        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): TelemetryConfigurationEvent {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryConfigurationEvent",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): TelemetryConfigurationEvent {
            try {
                val dd = Dd()
                val date = jsonObject.get("date").asLong
                val service = jsonObject.get("service").asString
                val source = Source.fromJson(jsonObject.get("source").asString)
                val version = jsonObject.get("version").asString
                val application = jsonObject.get("application")?.asJsonObject?.let {
                    Application.fromJsonObject(it)
                }
                val session = jsonObject.get("session")?.asJsonObject?.let {
                    Session.fromJsonObject(it)
                }
                val view = jsonObject.get("view")?.asJsonObject?.let {
                    View.fromJsonObject(it)
                }
                val action = jsonObject.get("action")?.asJsonObject?.let {
                    Action.fromJsonObject(it)
                }
                val experimentalFeatures =
                        jsonObject.get("experimental_features")?.asJsonArray?.let { jsonArray ->
                    val collection = ArrayList<String>(jsonArray.size())
                    jsonArray.forEach {
                        collection.add(it.asString)
                    }
                    collection
                }
                val telemetry = jsonObject.get("telemetry").asJsonObject.let {
                    Telemetry.fromJsonObject(it)
                }
                return TelemetryConfigurationEvent(dd, date, service, source, version, application,
                        session, view, action, experimentalFeatures, telemetry)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryConfigurationEvent",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryConfigurationEvent",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryConfigurationEvent",
                    e
                )
            }
        }
    }

    /**
     * Internal properties
     */
    public class Dd {
        public val formatVersion: Long = 2L

        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("format_version", formatVersion)
            return json
        }
    }

    /**
     * Application properties
     * @param id UUID of the application
     */
    public data class Application(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Application {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Application {
                try {
                    val id = jsonObject.get("id").asString
                    return Application(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }
        }
    }

    /**
     * Session properties
     * @param id UUID of the session
     */
    public data class Session(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Session {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Session {
                try {
                    val id = jsonObject.get("id").asString
                    return Session(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }
        }
    }

    /**
     * View properties
     * @param id UUID of the view
     */
    public data class View(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): View {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): View {
                try {
                    val id = jsonObject.get("id").asString
                    return View(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }
        }
    }

    /**
     * Action properties
     * @param id UUID of the action
     */
    public data class Action(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Action {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Action {
                try {
                    val id = jsonObject.get("id").asString
                    return Action(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }
        }
    }

    /**
     * The telemetry configuration information
     * @param configuration Configuration properties
     */
    public data class Telemetry(
        public val configuration: Configuration,
    ) {
        public val type: String = "configuration"

        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("type", type)
            json.add("configuration", configuration.toJson())
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Telemetry {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Telemetry {
                try {
                    val configuration = jsonObject.get("configuration").asJsonObject.let {
                        Configuration.fromJsonObject(it)
                    }
                    return Telemetry(configuration)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                }
            }
        }
    }

    /**
     * Configuration properties
     * @param sessionSampleRate The percentage of sessions tracked
     * @param telemetrySampleRate The percentage of telemetry events sent
     * @param telemetryConfigurationSampleRate The percentage of telemetry configuration events sent
     * after being sampled by telemetry_sample_rate
     * @param traceSampleRate The percentage of requests traced
     * @param premiumSampleRate The percentage of sessions with Browser RUM & Session Replay pricing
     * tracked (deprecated in favor of session_replay_sample_rate)
     * @param replaySampleRate The percentage of sessions with Browser RUM & Session Replay pricing
     * tracked (deprecated in favor of session_replay_sample_rate)
     * @param sessionReplaySampleRate The percentage of sessions with Browser RUM & Session Replay
     * pricing tracked
     * @param startSessionReplayRecordingManually Whether the session replay start is handled
     * manually
     * @param useProxy Whether a proxy configured is used
     * @param useBeforeSend Whether beforeSend callback function is used
     * @param silentMultipleInit Whether initialization fails silently if the SDK is already
     * initialized
     * @param trackSessionAcrossSubdomains Whether sessions across subdomains for the same site are
     * tracked
     * @param trackResources Whether resources are tracked
     * @param trackLongTask Whether long tasks are tracked
     * @param useCrossSiteSessionCookie Whether a secure cross-site session cookie is used
     * @param useSecureSessionCookie Whether a secure session cookie is used
     * @param actionNameAttribute Attribute to be used to name actions
     * @param useAllowedTracingOrigins Whether the allowed tracing origins list is used (deprecated
     * in favor of use_allowed_tracing_urls)
     * @param useAllowedTracingUrls Whether the allowed tracing urls list is used
     * @param selectedTracingPropagators A list of selected tracing propagators
     * @param defaultPrivacyLevel Session replay default privacy level
     * @param useExcludedActivityUrls Whether the request origins list to ignore when computing the
     * page activity is used
     * @param trackFrustrations Whether user frustrations are tracked
     * @param trackViewsManually Whether the RUM views creation is handled manually
     * @param trackInteractions Whether user actions are tracked (deprecated in favor of
     * track_user_interactions)
     * @param trackUserInteractions Whether user actions are tracked
     * @param forwardErrorsToLogs Whether console.error logs, uncaught exceptions and network errors
     * are tracked
     * @param useLocalEncryption Whether local encryption is used
     * @param viewTrackingStrategy View tracking strategy
     * @param trackBackgroundEvents Whether RUM events are tracked when the application is in
     * Background
     * @param mobileVitalsUpdatePeriod The period between each Mobile Vital sample (in milliseconds)
     * @param trackErrors Whether error monitoring & crash reporting is enabled for the source
     * platform
     * @param trackNetworkRequests Whether automatic collection of network requests is enabled
     * @param useTracing Whether tracing features are enabled
     * @param trackNativeViews Whether native views are tracked (for cross platform SDKs)
     * @param trackNativeErrors Whether native error monitoring & crash reporting is enabled (for
     * cross platform SDKs)
     * @param trackNativeLongTasks Whether long task tracking is performed automatically
     * @param trackCrossPlatformLongTasks Whether long task tracking is performed automatically for
     * cross platform SDKs
     * @param useFirstPartyHosts Whether the client has provided a list of first party hosts
     * @param initializationType The type of initialization the SDK used, in case multiple are
     * supported
     * @param trackFlutterPerformance Whether Flutter build and raster time tracking is enabled
     * @param batchSize The window duration for batches sent by the SDK (in milliseconds)
     * @param batchUploadFrequency The upload frequency of batches (in milliseconds)
     * @param reactVersion The version of React used in a ReactNative application
     * @param reactNativeVersion The version of ReactNative used in a ReactNative application
     * @param dartVersion The version of Dart used in a Flutter application
     */
    public data class Configuration(
        public val sessionSampleRate: Long? = null,
        public val telemetrySampleRate: Long? = null,
        public val telemetryConfigurationSampleRate: Long? = null,
        public val traceSampleRate: Long? = null,
        public val premiumSampleRate: Long? = null,
        public val replaySampleRate: Long? = null,
        public var sessionReplaySampleRate: Long? = null,
        public var startSessionReplayRecordingManually: Boolean? = null,
        public var useProxy: Boolean? = null,
        public val useBeforeSend: Boolean? = null,
        public val silentMultipleInit: Boolean? = null,
        public val trackSessionAcrossSubdomains: Boolean? = null,
        public var trackResources: Boolean? = null,
        public var trackLongTask: Boolean? = null,
        public val useCrossSiteSessionCookie: Boolean? = null,
        public val useSecureSessionCookie: Boolean? = null,
        public val actionNameAttribute: String? = null,
        public val useAllowedTracingOrigins: Boolean? = null,
        public val useAllowedTracingUrls: Boolean? = null,
        public val selectedTracingPropagators: List<SelectedTracingPropagator>? = null,
        public var defaultPrivacyLevel: String? = null,
        public val useExcludedActivityUrls: Boolean? = null,
        public var trackFrustrations: Boolean? = null,
        public var trackViewsManually: Boolean? = null,
        public var trackInteractions: Boolean? = null,
        public var trackUserInteractions: Boolean? = null,
        public val forwardErrorsToLogs: Boolean? = null,
        public val forwardConsoleLogs: List<String>? = null,
        public val forwardReports: List<String>? = null,
        public val useLocalEncryption: Boolean? = null,
        public val viewTrackingStrategy: ViewTrackingStrategy? = null,
        public var trackBackgroundEvents: Boolean? = null,
        public var mobileVitalsUpdatePeriod: Long? = null,
        public var trackErrors: Boolean? = null,
        public var trackNetworkRequests: Boolean? = null,
        public val useTracing: Boolean? = null,
        public var trackNativeViews: Boolean? = null,
        public var trackNativeErrors: Boolean? = null,
        public var trackNativeLongTasks: Boolean? = null,
        public var trackCrossPlatformLongTasks: Boolean? = null,
        public var useFirstPartyHosts: Boolean? = null,
        public var initializationType: String? = null,
        public var trackFlutterPerformance: Boolean? = null,
        public val batchSize: Long? = null,
        public val batchUploadFrequency: Long? = null,
        public var reactVersion: String? = null,
        public var reactNativeVersion: String? = null,
        public var dartVersion: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            sessionSampleRate?.let { sessionSampleRateNonNull ->
                json.addProperty("session_sample_rate", sessionSampleRateNonNull)
            }
            telemetrySampleRate?.let { telemetrySampleRateNonNull ->
                json.addProperty("telemetry_sample_rate", telemetrySampleRateNonNull)
            }
            telemetryConfigurationSampleRate?.let { telemetryConfigurationSampleRateNonNull ->
                json.addProperty("telemetry_configuration_sample_rate",
                        telemetryConfigurationSampleRateNonNull)
            }
            traceSampleRate?.let { traceSampleRateNonNull ->
                json.addProperty("trace_sample_rate", traceSampleRateNonNull)
            }
            premiumSampleRate?.let { premiumSampleRateNonNull ->
                json.addProperty("premium_sample_rate", premiumSampleRateNonNull)
            }
            replaySampleRate?.let { replaySampleRateNonNull ->
                json.addProperty("replay_sample_rate", replaySampleRateNonNull)
            }
            sessionReplaySampleRate?.let { sessionReplaySampleRateNonNull ->
                json.addProperty("session_replay_sample_rate", sessionReplaySampleRateNonNull)
            }
            startSessionReplayRecordingManually?.let { startSessionReplayRecordingManuallyNonNull ->
                json.addProperty("start_session_replay_recording_manually",
                        startSessionReplayRecordingManuallyNonNull)
            }
            useProxy?.let { useProxyNonNull ->
                json.addProperty("use_proxy", useProxyNonNull)
            }
            useBeforeSend?.let { useBeforeSendNonNull ->
                json.addProperty("use_before_send", useBeforeSendNonNull)
            }
            silentMultipleInit?.let { silentMultipleInitNonNull ->
                json.addProperty("silent_multiple_init", silentMultipleInitNonNull)
            }
            trackSessionAcrossSubdomains?.let { trackSessionAcrossSubdomainsNonNull ->
                json.addProperty("track_session_across_subdomains",
                        trackSessionAcrossSubdomainsNonNull)
            }
            trackResources?.let { trackResourcesNonNull ->
                json.addProperty("track_resources", trackResourcesNonNull)
            }
            trackLongTask?.let { trackLongTaskNonNull ->
                json.addProperty("track_long_task", trackLongTaskNonNull)
            }
            useCrossSiteSessionCookie?.let { useCrossSiteSessionCookieNonNull ->
                json.addProperty("use_cross_site_session_cookie", useCrossSiteSessionCookieNonNull)
            }
            useSecureSessionCookie?.let { useSecureSessionCookieNonNull ->
                json.addProperty("use_secure_session_cookie", useSecureSessionCookieNonNull)
            }
            actionNameAttribute?.let { actionNameAttributeNonNull ->
                json.addProperty("action_name_attribute", actionNameAttributeNonNull)
            }
            useAllowedTracingOrigins?.let { useAllowedTracingOriginsNonNull ->
                json.addProperty("use_allowed_tracing_origins", useAllowedTracingOriginsNonNull)
            }
            useAllowedTracingUrls?.let { useAllowedTracingUrlsNonNull ->
                json.addProperty("use_allowed_tracing_urls", useAllowedTracingUrlsNonNull)
            }
            selectedTracingPropagators?.let { selectedTracingPropagatorsNonNull ->
                val selectedTracingPropagatorsArray =
                        JsonArray(selectedTracingPropagatorsNonNull.size)
                selectedTracingPropagatorsNonNull.forEach {
                        selectedTracingPropagatorsArray.add(it.toJson()) }
                json.add("selected_tracing_propagators", selectedTracingPropagatorsArray)
            }
            defaultPrivacyLevel?.let { defaultPrivacyLevelNonNull ->
                json.addProperty("default_privacy_level", defaultPrivacyLevelNonNull)
            }
            useExcludedActivityUrls?.let { useExcludedActivityUrlsNonNull ->
                json.addProperty("use_excluded_activity_urls", useExcludedActivityUrlsNonNull)
            }
            trackFrustrations?.let { trackFrustrationsNonNull ->
                json.addProperty("track_frustrations", trackFrustrationsNonNull)
            }
            trackViewsManually?.let { trackViewsManuallyNonNull ->
                json.addProperty("track_views_manually", trackViewsManuallyNonNull)
            }
            trackInteractions?.let { trackInteractionsNonNull ->
                json.addProperty("track_interactions", trackInteractionsNonNull)
            }
            trackUserInteractions?.let { trackUserInteractionsNonNull ->
                json.addProperty("track_user_interactions", trackUserInteractionsNonNull)
            }
            forwardErrorsToLogs?.let { forwardErrorsToLogsNonNull ->
                json.addProperty("forward_errors_to_logs", forwardErrorsToLogsNonNull)
            }
            forwardConsoleLogs?.let { forwardConsoleLogsNonNull ->
                val forwardConsoleLogsArray = JsonArray(forwardConsoleLogsNonNull.size)
                forwardConsoleLogsNonNull.forEach { forwardConsoleLogsArray.add(it) }
                json.add("forward_console_logs", forwardConsoleLogsArray)
            }
            forwardReports?.let { forwardReportsNonNull ->
                val forwardReportsArray = JsonArray(forwardReportsNonNull.size)
                forwardReportsNonNull.forEach { forwardReportsArray.add(it) }
                json.add("forward_reports", forwardReportsArray)
            }
            useLocalEncryption?.let { useLocalEncryptionNonNull ->
                json.addProperty("use_local_encryption", useLocalEncryptionNonNull)
            }
            viewTrackingStrategy?.let { viewTrackingStrategyNonNull ->
                json.add("view_tracking_strategy", viewTrackingStrategyNonNull.toJson())
            }
            trackBackgroundEvents?.let { trackBackgroundEventsNonNull ->
                json.addProperty("track_background_events", trackBackgroundEventsNonNull)
            }
            mobileVitalsUpdatePeriod?.let { mobileVitalsUpdatePeriodNonNull ->
                json.addProperty("mobile_vitals_update_period", mobileVitalsUpdatePeriodNonNull)
            }
            trackErrors?.let { trackErrorsNonNull ->
                json.addProperty("track_errors", trackErrorsNonNull)
            }
            trackNetworkRequests?.let { trackNetworkRequestsNonNull ->
                json.addProperty("track_network_requests", trackNetworkRequestsNonNull)
            }
            useTracing?.let { useTracingNonNull ->
                json.addProperty("use_tracing", useTracingNonNull)
            }
            trackNativeViews?.let { trackNativeViewsNonNull ->
                json.addProperty("track_native_views", trackNativeViewsNonNull)
            }
            trackNativeErrors?.let { trackNativeErrorsNonNull ->
                json.addProperty("track_native_errors", trackNativeErrorsNonNull)
            }
            trackNativeLongTasks?.let { trackNativeLongTasksNonNull ->
                json.addProperty("track_native_long_tasks", trackNativeLongTasksNonNull)
            }
            trackCrossPlatformLongTasks?.let { trackCrossPlatformLongTasksNonNull ->
                json.addProperty("track_cross_platform_long_tasks",
                        trackCrossPlatformLongTasksNonNull)
            }
            useFirstPartyHosts?.let { useFirstPartyHostsNonNull ->
                json.addProperty("use_first_party_hosts", useFirstPartyHostsNonNull)
            }
            initializationType?.let { initializationTypeNonNull ->
                json.addProperty("initialization_type", initializationTypeNonNull)
            }
            trackFlutterPerformance?.let { trackFlutterPerformanceNonNull ->
                json.addProperty("track_flutter_performance", trackFlutterPerformanceNonNull)
            }
            batchSize?.let { batchSizeNonNull ->
                json.addProperty("batch_size", batchSizeNonNull)
            }
            batchUploadFrequency?.let { batchUploadFrequencyNonNull ->
                json.addProperty("batch_upload_frequency", batchUploadFrequencyNonNull)
            }
            reactVersion?.let { reactVersionNonNull ->
                json.addProperty("react_version", reactVersionNonNull)
            }
            reactNativeVersion?.let { reactNativeVersionNonNull ->
                json.addProperty("react_native_version", reactNativeVersionNonNull)
            }
            dartVersion?.let { dartVersionNonNull ->
                json.addProperty("dart_version", dartVersionNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Configuration {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Configuration {
                try {
                    val sessionSampleRate = jsonObject.get("session_sample_rate")?.asLong
                    val telemetrySampleRate = jsonObject.get("telemetry_sample_rate")?.asLong
                    val telemetryConfigurationSampleRate =
                            jsonObject.get("telemetry_configuration_sample_rate")?.asLong
                    val traceSampleRate = jsonObject.get("trace_sample_rate")?.asLong
                    val premiumSampleRate = jsonObject.get("premium_sample_rate")?.asLong
                    val replaySampleRate = jsonObject.get("replay_sample_rate")?.asLong
                    val sessionReplaySampleRate =
                            jsonObject.get("session_replay_sample_rate")?.asLong
                    val startSessionReplayRecordingManually =
                            jsonObject.get("start_session_replay_recording_manually")?.asBoolean
                    val useProxy = jsonObject.get("use_proxy")?.asBoolean
                    val useBeforeSend = jsonObject.get("use_before_send")?.asBoolean
                    val silentMultipleInit = jsonObject.get("silent_multiple_init")?.asBoolean
                    val trackSessionAcrossSubdomains =
                            jsonObject.get("track_session_across_subdomains")?.asBoolean
                    val trackResources = jsonObject.get("track_resources")?.asBoolean
                    val trackLongTask = jsonObject.get("track_long_task")?.asBoolean
                    val useCrossSiteSessionCookie =
                            jsonObject.get("use_cross_site_session_cookie")?.asBoolean
                    val useSecureSessionCookie =
                            jsonObject.get("use_secure_session_cookie")?.asBoolean
                    val actionNameAttribute = jsonObject.get("action_name_attribute")?.asString
                    val useAllowedTracingOrigins =
                            jsonObject.get("use_allowed_tracing_origins")?.asBoolean
                    val useAllowedTracingUrls =
                            jsonObject.get("use_allowed_tracing_urls")?.asBoolean
                    val selectedTracingPropagators =
                            jsonObject.get("selected_tracing_propagators")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<SelectedTracingPropagator>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(SelectedTracingPropagator.fromJson(it.asString))
                        }
                        collection
                    }
                    val defaultPrivacyLevel = jsonObject.get("default_privacy_level")?.asString
                    val useExcludedActivityUrls =
                            jsonObject.get("use_excluded_activity_urls")?.asBoolean
                    val trackFrustrations = jsonObject.get("track_frustrations")?.asBoolean
                    val trackViewsManually = jsonObject.get("track_views_manually")?.asBoolean
                    val trackInteractions = jsonObject.get("track_interactions")?.asBoolean
                    val trackUserInteractions = jsonObject.get("track_user_interactions")?.asBoolean
                    val forwardErrorsToLogs = jsonObject.get("forward_errors_to_logs")?.asBoolean
                    val forwardConsoleLogs =
                            jsonObject.get("forward_console_logs")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<String>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(it.asString)
                        }
                        collection
                    }
                    val forwardReports =
                            jsonObject.get("forward_reports")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<String>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(it.asString)
                        }
                        collection
                    }
                    val useLocalEncryption = jsonObject.get("use_local_encryption")?.asBoolean
                    val viewTrackingStrategy =
                            jsonObject.get("view_tracking_strategy")?.asString?.let {
                        ViewTrackingStrategy.fromJson(it)
                    }
                    val trackBackgroundEvents = jsonObject.get("track_background_events")?.asBoolean
                    val mobileVitalsUpdatePeriod =
                            jsonObject.get("mobile_vitals_update_period")?.asLong
                    val trackErrors = jsonObject.get("track_errors")?.asBoolean
                    val trackNetworkRequests = jsonObject.get("track_network_requests")?.asBoolean
                    val useTracing = jsonObject.get("use_tracing")?.asBoolean
                    val trackNativeViews = jsonObject.get("track_native_views")?.asBoolean
                    val trackNativeErrors = jsonObject.get("track_native_errors")?.asBoolean
                    val trackNativeLongTasks = jsonObject.get("track_native_long_tasks")?.asBoolean
                    val trackCrossPlatformLongTasks =
                            jsonObject.get("track_cross_platform_long_tasks")?.asBoolean
                    val useFirstPartyHosts = jsonObject.get("use_first_party_hosts")?.asBoolean
                    val initializationType = jsonObject.get("initialization_type")?.asString
                    val trackFlutterPerformance =
                            jsonObject.get("track_flutter_performance")?.asBoolean
                    val batchSize = jsonObject.get("batch_size")?.asLong
                    val batchUploadFrequency = jsonObject.get("batch_upload_frequency")?.asLong
                    val reactVersion = jsonObject.get("react_version")?.asString
                    val reactNativeVersion = jsonObject.get("react_native_version")?.asString
                    val dartVersion = jsonObject.get("dart_version")?.asString
                    return Configuration(sessionSampleRate, telemetrySampleRate,
                            telemetryConfigurationSampleRate, traceSampleRate, premiumSampleRate,
                            replaySampleRate, sessionReplaySampleRate,
                            startSessionReplayRecordingManually, useProxy, useBeforeSend,
                            silentMultipleInit, trackSessionAcrossSubdomains, trackResources,
                            trackLongTask, useCrossSiteSessionCookie, useSecureSessionCookie,
                            actionNameAttribute, useAllowedTracingOrigins, useAllowedTracingUrls,
                            selectedTracingPropagators, defaultPrivacyLevel,
                            useExcludedActivityUrls, trackFrustrations, trackViewsManually,
                            trackInteractions, trackUserInteractions, forwardErrorsToLogs,
                            forwardConsoleLogs, forwardReports, useLocalEncryption,
                            viewTrackingStrategy, trackBackgroundEvents, mobileVitalsUpdatePeriod,
                            trackErrors, trackNetworkRequests, useTracing, trackNativeViews,
                            trackNativeErrors, trackNativeLongTasks, trackCrossPlatformLongTasks,
                            useFirstPartyHosts, initializationType, trackFlutterPerformance,
                            batchSize, batchUploadFrequency, reactVersion, reactNativeVersion,
                            dartVersion)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Configuration",
                        e
                    )
                }
            }
        }
    }

    /**
     * The source of this event
     */
    public enum class Source(
        private val jsonValue: String,
    ) {
        ANDROID("android"),
        IOS("ios"),
        BROWSER("browser"),
        FLUTTER("flutter"),
        REACT_NATIVE("react-native"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Source = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    public enum class SelectedTracingPropagator(
        private val jsonValue: String,
    ) {
        DATADOG("datadog"),
        B3("b3"),
        B3MULTI("b3multi"),
        TRACECONTEXT("tracecontext"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): SelectedTracingPropagator = values().first {
                it.jsonValue == jsonString
            }
        }
    }

    /**
     * View tracking strategy
     */
    public enum class ViewTrackingStrategy(
        private val jsonValue: String,
    ) {
        ACTIVITYVIEWTRACKINGSTRATEGY("ActivityViewTrackingStrategy"),
        FRAGMENTVIEWTRACKINGSTRATEGY("FragmentViewTrackingStrategy"),
        MIXEDVIEWTRACKINGSTRATEGY("MixedViewTrackingStrategy"),
        NAVIGATIONVIEWTRACKINGSTRATEGY("NavigationViewTrackingStrategy"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): ViewTrackingStrategy = values().first {
                it.jsonValue == jsonString
            }
        }
    }
}
