/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.dse.graph;

import com.datastax.driver.core.ConsistencyLevel;
import com.datastax.driver.core.ProtocolVersion;
import com.datastax.driver.core.SimpleStatement;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.TypeCodec;
import com.datastax.driver.dse.DseSessionHook;
import com.datastax.driver.dse.graph.GraphOptions;
import com.datastax.driver.dse.graph.GraphStatement;
import com.datastax.driver.dse.graph.PayloadHelper;
import com.datastax.driver.dse.graph.SimpleGraphStatement;
import com.google.common.collect.ImmutableMap;
import java.nio.ByteBuffer;
import java.util.Map;
import org.assertj.core.api.Assertions;
import org.testng.annotations.Test;

public class GraphOptionsPayloadTest {
    private static final String GRAPH_CONFIG_PREFIX = "cfg.";

    @Test(groups={"unit"})
    public void should_use_default_options_when_none_set() {
        GraphOptions graphOptions = new GraphOptions();
        Map<String, ByteBuffer> expectedPayload = this.buildPayloadFromProperties("gremlin-groovy", null, "g", null, null);
        Map resultPayload = graphOptions.buildPayloadWithDefaults((GraphStatement)new SimpleGraphStatement(""));
        Assertions.assertThat((Map)resultPayload).isEqualTo(expectedPayload);
    }

    @Test(groups={"unit"})
    public void should_use_cluster_options_set() {
        GraphOptions graphOptions = new GraphOptions();
        graphOptions.setGraphLanguage("language1");
        graphOptions.setGraphName("name1");
        graphOptions.setGraphSource("source1");
        graphOptions.setGraphReadConsistencyLevel(ConsistencyLevel.LOCAL_QUORUM);
        graphOptions.setGraphWriteConsistencyLevel(ConsistencyLevel.ALL);
        Map<String, ByteBuffer> expectedPayload = this.buildPayloadFromProperties("language1", "name1", "source1", ConsistencyLevel.LOCAL_QUORUM, ConsistencyLevel.ALL);
        Map resultPayload = graphOptions.buildPayloadWithDefaults((GraphStatement)new SimpleGraphStatement(""));
        Assertions.assertThat((Map)resultPayload).isEqualTo(expectedPayload);
    }

    @Test(groups={"unit"})
    public void should_use_statement_options_over_cluster_options() {
        GraphOptions graphOptions = new GraphOptions();
        graphOptions.setGraphLanguage("language1");
        graphOptions.setGraphName("name1");
        graphOptions.setGraphSource("source1");
        graphOptions.setGraphReadConsistencyLevel(ConsistencyLevel.LOCAL_QUORUM);
        graphOptions.setGraphWriteConsistencyLevel(ConsistencyLevel.ALL);
        SimpleGraphStatement simpleGraphStatement = new SimpleGraphStatement("");
        simpleGraphStatement.setGraphLanguage("language2");
        simpleGraphStatement.setGraphName("name2");
        simpleGraphStatement.setGraphSource("source2");
        simpleGraphStatement.setGraphReadConsistencyLevel(ConsistencyLevel.ANY);
        simpleGraphStatement.setGraphWriteConsistencyLevel(ConsistencyLevel.EACH_QUORUM);
        Map<String, ByteBuffer> expectedPayload = this.buildPayloadFromStatement((GraphStatement)simpleGraphStatement);
        Map resultPayload = graphOptions.buildPayloadWithDefaults((GraphStatement)simpleGraphStatement);
        Assertions.assertThat((Map)resultPayload).isEqualTo(expectedPayload);
    }

    @Test(groups={"unit"}, expectedExceptions={NullPointerException.class}, expectedExceptionsMessageRegExp="graphLanguage cannot be null")
    public void should_not_allow_null_on_graph_language_on_cluster() {
        GraphOptions graphOptions = new GraphOptions();
        graphOptions.setGraphLanguage(null);
    }

    @Test(groups={"unit"}, expectedExceptions={NullPointerException.class}, expectedExceptionsMessageRegExp="graphSource cannot be null")
    public void should_not_allow_null_on_graph_source_on_cluster() {
        GraphOptions graphOptions = new GraphOptions();
        graphOptions.setGraphSource(null);
    }

    @Test(groups={"unit"})
    public void should_allow_null_graph_name_on_cluster() {
        GraphOptions graphOptions = new GraphOptions();
        graphOptions.setGraphLanguage("language1");
        graphOptions.setGraphSource("source1");
        graphOptions.setGraphName(null);
        Map<String, ByteBuffer> expectedPayload = this.buildPayloadFromProperties("language1", null, "source1", null, null);
        Map resultPayload = graphOptions.buildPayloadWithDefaults((GraphStatement)new SimpleGraphStatement(""));
        Assertions.assertThat((Map)resultPayload).isEqualTo(expectedPayload);
    }

    @Test(groups={"unit"})
    public void should_force_no_graph_name_if_statement_is_a_system_query() {
        GraphOptions graphOptions = new GraphOptions();
        graphOptions.setGraphLanguage("language1");
        graphOptions.setGraphName("name1");
        graphOptions.setGraphSource("source1");
        SimpleGraphStatement simpleGraphStatement = new SimpleGraphStatement("");
        simpleGraphStatement.setSystemQuery();
        Map<String, ByteBuffer> expectedPayload = this.buildPayloadFromProperties("language1", null, "source1", null, null);
        Map resultPayload = graphOptions.buildPayloadWithDefaults((GraphStatement)simpleGraphStatement);
        Assertions.assertThat((Map)resultPayload).isEqualTo(expectedPayload);
    }

    @Test(groups={"unit"})
    public void should_use_native_consistency_and_timestamp() {
        ConsistencyLevel desiredCL = ConsistencyLevel.ALL;
        long desiredTimestamp = 12L;
        SimpleGraphStatement sgt = new SimpleGraphStatement("query");
        sgt.setConsistencyLevel(desiredCL);
        sgt.setDefaultTimestamp(desiredTimestamp);
        SimpleStatement st = sgt.unwrap();
        Assertions.assertThat((Comparable)st.getConsistencyLevel()).isEqualTo((Object)desiredCL);
        Assertions.assertThat((long)st.getDefaultTimestamp()).isEqualTo(desiredTimestamp);
        sgt.set("a", (Object)"a");
        st = sgt.unwrap();
        Assertions.assertThat((Comparable)st.getConsistencyLevel()).isEqualTo((Object)desiredCL);
        Assertions.assertThat((long)st.getDefaultTimestamp()).isEqualTo(desiredTimestamp);
    }

    @Test(groups={"unit"})
    public void should_set_default_timeout_for_graph_queries() {
        GraphOptions graphOptions = new GraphOptions();
        SimpleGraphStatement simpleGraphStatement = new SimpleGraphStatement("");
        Statement st = DseSessionHook.callGenerateCoreStatement(graphOptions, (GraphStatement)simpleGraphStatement);
        Assertions.assertThat((int)st.getReadTimeoutMillis()).isEqualTo(0);
    }

    @Test(groups={"unit"})
    public void should_set_read_timeout_from_statement_over_cluster() {
        int desiredClusterTimeout = 4000;
        int desiredStatementTimeout = 50000;
        GraphOptions graphOptions = new GraphOptions();
        graphOptions.setReadTimeoutMillis(desiredClusterTimeout);
        SimpleGraphStatement simpleGraphStatement = new SimpleGraphStatement("");
        Statement st = DseSessionHook.callGenerateCoreStatement(graphOptions, (GraphStatement)simpleGraphStatement);
        Assertions.assertThat((int)st.getReadTimeoutMillis()).isEqualTo(desiredClusterTimeout);
        simpleGraphStatement.setReadTimeoutMillis(desiredStatementTimeout);
        st = DseSessionHook.callGenerateCoreStatement(graphOptions, (GraphStatement)simpleGraphStatement);
        Assertions.assertThat((int)st.getReadTimeoutMillis()).isEqualTo(desiredStatementTimeout);
    }

    @Test
    public void should_use_graph_internal_options() {
        SimpleGraphStatement simpleGraphStatement = new SimpleGraphStatement("");
        String configKey = "external_vertex_verify";
        simpleGraphStatement.setGraphInternalOption("cfg.external_vertex_verify", "true");
        Statement st = DseSessionHook.callGenerateCoreStatement(new GraphOptions(), (GraphStatement)simpleGraphStatement);
        Assertions.assertThat((String)simpleGraphStatement.getGraphInternalOption("cfg.external_vertex_verify")).isEqualTo((Object)"true");
        Assertions.assertThat((String)((String)TypeCodec.varchar().deserialize((ByteBuffer)st.getOutgoingPayload().get(GRAPH_CONFIG_PREFIX + configKey), ProtocolVersion.NEWEST_SUPPORTED))).isEqualTo((Object)"true");
        simpleGraphStatement.setGraphInternalOption("cfg.external_vertex_verify", "false");
        st = DseSessionHook.callGenerateCoreStatement(new GraphOptions(), (GraphStatement)simpleGraphStatement);
        Assertions.assertThat((String)simpleGraphStatement.getGraphInternalOption("cfg.external_vertex_verify")).isEqualTo((Object)"false");
        Assertions.assertThat((String)((String)TypeCodec.varchar().deserialize((ByteBuffer)st.getOutgoingPayload().get(GRAPH_CONFIG_PREFIX + configKey), ProtocolVersion.NEWEST_SUPPORTED))).isEqualTo((Object)"false");
        simpleGraphStatement.setGraphInternalOption("cfg.external_vertex_verify", null);
        st = DseSessionHook.callGenerateCoreStatement(new GraphOptions(), (GraphStatement)simpleGraphStatement);
        Assertions.assertThat((String)simpleGraphStatement.getGraphInternalOption("cfg.external_vertex_verify")).isNull();
        Assertions.assertThat((String)((String)TypeCodec.varchar().deserialize((ByteBuffer)st.getOutgoingPayload().get(GRAPH_CONFIG_PREFIX + configKey), ProtocolVersion.NEWEST_SUPPORTED))).isNull();
    }

    @Test
    public void should_propagate_idempotence_from_graph_statement() {
        SimpleGraphStatement graphStatement = new SimpleGraphStatement("");
        Statement coreStatement = DseSessionHook.callGenerateCoreStatement(new GraphOptions(), (GraphStatement)graphStatement);
        Assertions.assertThat((Boolean)coreStatement.isIdempotent()).isNull();
        graphStatement.setIdempotent(false);
        coreStatement = DseSessionHook.callGenerateCoreStatement(new GraphOptions(), (GraphStatement)graphStatement);
        Assertions.assertThat((Boolean)coreStatement.isIdempotent()).isFalse();
        graphStatement.setIdempotent(true);
        coreStatement = DseSessionHook.callGenerateCoreStatement(new GraphOptions(), (GraphStatement)graphStatement);
        Assertions.assertThat((Boolean)coreStatement.isIdempotent()).isTrue();
    }

    private Map<String, ByteBuffer> buildPayloadFromStatement(GraphStatement graphStatement) {
        return ImmutableMap.of((Object)"graph-language", (Object)PayloadHelper.asBytes((String)graphStatement.getGraphLanguage()), (Object)"graph-name", (Object)PayloadHelper.asBytes((String)graphStatement.getGraphName()), (Object)"graph-source", (Object)PayloadHelper.asBytes((String)graphStatement.getGraphSource()), (Object)"graph-read-consistency", (Object)PayloadHelper.asBytes((String)graphStatement.getGraphReadConsistencyLevel().name()), (Object)"graph-write-consistency", (Object)PayloadHelper.asBytes((String)graphStatement.getGraphWriteConsistencyLevel().name()));
    }

    private Map<String, ByteBuffer> buildPayloadFromProperties(String graphLanguage, String graphName, String graphSource, ConsistencyLevel readCL, ConsistencyLevel writeCL) {
        ImmutableMap.Builder builder = ImmutableMap.builder();
        if (graphLanguage != null) {
            builder.put((Object)"graph-language", (Object)PayloadHelper.asBytes((String)graphLanguage));
        }
        if (graphName != null) {
            builder.put((Object)"graph-name", (Object)PayloadHelper.asBytes((String)graphName));
        }
        if (graphSource != null) {
            builder.put((Object)"graph-source", (Object)PayloadHelper.asBytes((String)graphSource));
        }
        if (readCL != null) {
            builder.put((Object)"graph-read-consistency", (Object)PayloadHelper.asBytes((String)readCL.name()));
        }
        if (writeCL != null) {
            builder.put((Object)"graph-write-consistency", (Object)PayloadHelper.asBytes((String)writeCL.name()));
        }
        return builder.build();
    }
}

