/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.dse.driver.api.core;

import com.datastax.dse.driver.api.core.config.DseDriverConfigLoader;
import com.datastax.dse.driver.api.core.type.codec.DseTypeCodecs;
import com.datastax.dse.driver.internal.core.config.typesafe.DefaultDseDriverConfigLoader;
import com.datastax.dse.driver.internal.core.context.DseDriverContext;
import com.datastax.dse.driver.internal.core.session.DefaultDseSession;
import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.config.DriverConfigLoader;
import com.datastax.oss.driver.api.core.context.DriverContext;
import com.datastax.oss.driver.api.core.metadata.Node;
import com.datastax.oss.driver.api.core.metadata.NodeStateListener;
import com.datastax.oss.driver.api.core.metadata.schema.SchemaChangeListener;
import com.datastax.oss.driver.api.core.session.SessionBuilder;
import com.datastax.oss.driver.api.core.tracker.RequestTracker;
import com.datastax.oss.driver.api.core.type.codec.TypeCodec;
import com.datastax.oss.driver.api.core.uuid.Uuids;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Predicate;
import net.jcip.annotations.NotThreadSafe;

@NotThreadSafe
public class DseSessionBuilder extends SessionBuilder<DseSessionBuilder, DseSession> {

  // Startup options
  private UUID startupClientId;
  private String startupApplicationName;
  private String startupApplicationVersion;

  /**
   * A unique identifier for the created session.
   *
   * <p>It will be sent in the {@code STARTUP} protocol message for each new connection established
   * by the driver, and may be used by future DSE versions for monitoring purposes.
   *
   * <p>If you don't call this method, the driver will generate an identifier with {@link
   * Uuids#random()}.
   */
  @NonNull
  public DseSessionBuilder withClientId(@Nullable UUID clientId) {
    this.startupClientId = clientId;
    return this;
  }

  /**
   * The name of the application using the created session.
   *
   * <p>It will be sent in the {@code STARTUP} protocol message for each new connection established
   * by the driver, and may be used by future DSE versions for monitoring purposes.
   *
   * <p>This can also be defined in the driver configuration with the option {@code
   * basic.application.name}; if you specify both, this method takes precedence and the
   * configuration option will be ignored.
   */
  @NonNull
  public DseSessionBuilder withApplicationName(@Nullable String applicationName) {
    this.startupApplicationName = applicationName;
    return this;
  }

  /**
   * The version of the application using the created session.
   *
   * <p>It will be sent in the {@code STARTUP} protocol message for each new connection established
   * by the driver, and may be used by future DSE versions for monitoring purposes.
   *
   * <p>This can also be defined in the driver configuration with the option {@code
   * basic.application.version}; if you specify both, this method takes precedence and the
   * configuration option will be ignored.
   */
  @NonNull
  public DseSessionBuilder withApplicationVersion(@Nullable String applicationVersion) {
    this.startupApplicationVersion = applicationVersion;
    return this;
  }

  /**
   * Sets the configuration loader to use.
   *
   * <p>Note that this loader must produce a configuration that includes the DSE-specific options:
   * if you're using one of the built-in implementations provided by the driver, use the static
   * factory methods from {@link DseDriverConfigLoader} (<b>not</b> the ones from {@link
   * DriverConfigLoader}).
   *
   * <p>If you don't call this method, the builder will use the default implementation, based on the
   * Typesafe config library. More precisely, configuration properties are loaded and merged from
   * the following (first-listed are higher priority):
   *
   * <ul>
   *   <li>system properties
   *   <li>{@code application.conf} (all resources on classpath with this name)
   *   <li>{@code application.json} (all resources on classpath with this name)
   *   <li>{@code application.properties} (all resources on classpath with this name)
   *   <li>{@code dse-reference.conf} (all resources on classpath with this name). In particular,
   *       this will load the {@code dse-reference.conf} included in the core DSE driver JAR, that
   *       defines default options for all DSE-specific mandatory options.
   *   <li>{@code reference.conf} (all resources on classpath with this name). In particular, this
   *       will load the {@code reference.conf} included in the core driver JAR, that defines
   *       default options for all mandatory options.
   * </ul>
   *
   * The resulting configuration is expected to contain a {@code datastax-java-driver} section.
   *
   * <p>This default loader will honor the reload interval defined by the option {@code
   * basic.config-reload-interval}.
   *
   * @see <a href="https://github.com/typesafehub/config#standard-behavior">Typesafe config's
   *     standard loading behavior</a>
   */
  @NonNull
  @Override
  public DseSessionBuilder withConfigLoader(@Nullable DriverConfigLoader configLoader) {
    // overridden only to customize the javadocs
    return super.withConfigLoader(configLoader);
  }

  @Override
  protected DriverContext buildContext(
      DriverConfigLoader configLoader,
      List<TypeCodec<?>> typeCodecs,
      NodeStateListener nodeStateListener,
      SchemaChangeListener schemaChangeListener,
      RequestTracker requestTracker,
      Map<String, String> localDatacenters,
      Map<String, Predicate<Node>> nodeFilters,
      ClassLoader classLoader) {
    typeCodecs.add(DseTypeCodecs.LINE_STRING);
    typeCodecs.add(DseTypeCodecs.POINT);
    typeCodecs.add(DseTypeCodecs.POLYGON);
    typeCodecs.add(DseTypeCodecs.DATE_RANGE);
    return new DseDriverContext(
        configLoader,
        typeCodecs,
        nodeStateListener,
        schemaChangeListener,
        requestTracker,
        localDatacenters,
        nodeFilters,
        classLoader,
        startupClientId,
        startupApplicationName,
        startupApplicationVersion);
  }

  @NonNull
  @Override
  protected DriverConfigLoader defaultConfigLoader() {
    return new DefaultDseDriverConfigLoader();
  }

  @NonNull
  @Override
  protected DseSession wrap(@NonNull CqlSession defaultSession) {
    return new DefaultDseSession(defaultSession);
  }
}
