/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.dse.driver.api.core.cql.reactive;

import com.datastax.oss.driver.api.core.cql.ColumnDefinitions;
import com.datastax.oss.driver.api.core.cql.ExecutionInfo;
import com.datastax.oss.driver.api.core.cql.Statement;
import edu.umd.cs.findbugs.annotations.NonNull;
import org.reactivestreams.Publisher;

/**
 * A {@link Publisher} of {@link ReactiveRow}s returned by a {@link ReactiveSession}.
 *
 * <p>By default, all implementations returned by the driver are cold, unicast, single-subscriber
 * only publishers. In other words, <em>they do not support multiple subscriptions</em>; consider
 * caching the results produced by such publishers if you need to consume them by more than one
 * downstream subscriber.
 *
 * <p>Also, note that reactive result sets may emit items to their subscribers on an internal driver
 * IO thread. Subscriber implementors are encouraged to abide by <a
 * href="https://github.com/reactive-streams/reactive-streams-jvm#2.2">Reactive Streams
 * Specification rule 2.2</a> and avoid performing heavy computations or blocking calls inside
 * {@link org.reactivestreams.Subscriber#onNext(Object) onNext} calls, as doing so could slow down
 * the driver and impact performance. Instead, they should asynchronously dispatch received signals
 * to their processing logic.
 *
 * <p>This interface exists mainly to expose useful information about {@linkplain
 * #getExecutionInfos() request execution} and {@linkplain #getColumnDefinitions() query metadata}.
 * This is particularly convenient for queries that do not return rows; for queries that do return
 * rows, it is also possible, and oftentimes easier, to access that same information {@linkplain
 * ReactiveRow at row level}.
 *
 * @see ReactiveSession#executeReactive(String)
 * @see ReactiveSession#executeReactive(Statement)
 * @see ReactiveRow
 */
public interface ReactiveResultSet extends Publisher<ReactiveRow> {

  /**
   * Returns metadata about the {@linkplain ColumnDefinitions columns} contained in this result set.
   *
   * <p>This publisher emits exactly one item as soon as the first response arrives, then completes.
   * If the query execution fails <em>within the first request-response cycle</em>, then this
   * publisher will fail with the same error; however if the error happens <em>after the first
   * response</em>, then this publisher will be already completed and will not acknowledge that
   * error in any way.
   *
   * <p>By default, publishers returned by this method do not support multiple subscriptions.
   *
   * @see ReactiveRow#getColumnDefinitions()
   */
  @NonNull
  Publisher<? extends ColumnDefinitions> getColumnDefinitions();

  /**
   * Returns {@linkplain ExecutionInfo information about the execution} of all requests that have
   * been performed so far to assemble this result set.
   *
   * <p>If the query is not paged, this publisher will emit exactly one item as soon as the response
   * arrives, then complete. If the query is paged, it will emit multiple items, one per page; then
   * it will complete when the last page arrives. If the query execution fails, then this publisher
   * will fail with the same error.
   *
   * <p>By default, publishers returned by this method do not support multiple subscriptions.
   *
   * @see ReactiveRow#getExecutionInfo()
   */
  @NonNull
  Publisher<? extends ExecutionInfo> getExecutionInfos();

  /**
   * If the query that produced this result was a conditional update, indicates whether it was
   * successfully applied.
   *
   * <p>This publisher emits exactly one item as soon as the first response arrives, then completes.
   * If the query execution fails <em>within the first request-response cycle</em>, then this
   * publisher will fail with the same error; however if the error happens <em>after the first
   * response</em>, then this publisher will be already completed and will not acknowledge that
   * error in any way.
   *
   * <p>By default, publishers returned by this method do not support multiple subscriptions.
   *
   * <p>For consistency, this method always returns {@code true} for non-conditional queries
   * (although there is no reason to call the method in that case). This is also the case for
   * conditional DDL statements ({@code CREATE KEYSPACE... IF NOT EXISTS}, {@code CREATE TABLE... IF
   * NOT EXISTS}), for which Cassandra doesn't return an {@code [applied]} column.
   *
   * <p>Note that, for versions of Cassandra strictly lower than 2.1.0-rc2, a server-side bug (<a
   * href="https://issues.apache.org/jira/browse/CASSANDRA-7337">CASSANDRA-7337</a>) causes this
   * method to always return {@code true} for batches containing conditional queries.
   *
   * @see ReactiveRow#wasApplied()
   */
  @NonNull
  Publisher<Boolean> wasApplied();
}
