/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.dse.driver.internal.core.auth;

import com.datastax.dse.driver.api.core.config.DseDriverOption;
import com.datastax.oss.driver.api.core.auth.AuthProvider;
import com.datastax.oss.driver.api.core.auth.AuthenticationException;
import com.datastax.oss.driver.api.core.auth.Authenticator;
import com.datastax.oss.driver.api.core.config.DriverExecutionProfile;
import com.datastax.oss.driver.api.core.context.DriverContext;
import com.datastax.oss.driver.api.core.metadata.EndPoint;
import com.datastax.oss.driver.shaded.guava.common.base.Charsets;
import com.datastax.oss.driver.shaded.guava.common.collect.ImmutableMap;
import com.datastax.oss.protocol.internal.util.Bytes;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.HashMap;
import java.util.Map;
import javax.security.auth.Subject;
import javax.security.auth.login.AppConfigurationEntry;
import javax.security.auth.login.Configuration;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;
import javax.security.sasl.Sasl;
import javax.security.sasl.SaslClient;
import javax.security.sasl.SaslException;
import net.jcip.annotations.ThreadSafe;

/**
 * {@link AuthProvider} that provides GSSAPI authenticator instances for clients to connect to DSE
 * clusters secured with {@code DseAuthenticator}.
 *
 * <p>To activate this provider an {@code auth-provider} section must be included in the driver
 * configuration, for example:
 *
 * <pre>
 * dse-java-driver {
 *  auth-provider {
 *      class = com.datastax.dse.driver.internal.core.auth.DseGssApiAuthProvider
 *      login-configuration {
 *          principal = "user principal here ex cassandra@DATASTAX.COM"
 *          useKeyTab = "true"
 *          refreshKrb5Config = "true"
 *          keyTab = "Path to keytab file here"
 *      }
 *   }
 * }
 * </pre>
 *
 * <h2>Kerberos Authentication</h2>
 *
 * Keytab and ticket cache settings are specified using a standard JAAS configuration file. The
 * location of the file can be set using the <code>java.security.auth.login.config</code> system
 * property or by adding a <code>login.config.url.n</code> entry in the <code>java.security</code>
 * properties file. Alternatively a login-configuration section can be included in the driver
 * configuration.
 *
 * <p>See the following documents for further details:
 *
 * <ol>
 *   <li><a
 *       href="https://docs.oracle.com/javase/6/docs/technotes/guides/security/jgss/tutorials/LoginConfigFile.html">JAAS
 *       Login Configuration File</a>;
 *   <li><a
 *       href="https://docs.oracle.com/javase/6/docs/jre/api/security/jaas/spec/com/sun/security/auth/module/Krb5LoginModule.html">Krb5LoginModule
 *       options</a>;
 *   <li><a
 *       href="http://docs.oracle.com/javase/6/docs/technotes/guides/security/jaas/tutorials/GeneralAcnOnly.html">JAAS
 *       Authentication Tutorial</a> for more on JAAS in general.
 * </ol>
 *
 * <h3>Authentication using ticket cache</h3>
 *
 * Run <code>kinit</code> to obtain a ticket and populate the cache before connecting. JAAS config:
 *
 * <pre>
 * DseClient {
 *   com.sun.security.auth.module.Krb5LoginModule required
 *     useTicketCache=true
 *     renewTGT=true;
 * };
 * </pre>
 *
 * <h3>Authentication using a keytab file</h3>
 *
 * To enable authentication using a keytab file, specify its location on disk. If your keytab
 * contains more than one principal key, you should also specify which one to select. This
 * information can also be specified in the driver config, under the login-configuration section.
 *
 * <pre>
 * DseClient {
 *     com.sun.security.auth.module.Krb5LoginModule required
 *       useKeyTab=true
 *       keyTab="/path/to/file.keytab"
 *       principal="user@MYDOMAIN.COM";
 * };
 * </pre>
 *
 * <h2>Specifying SASL protocol name</h2>
 *
 * The SASL protocol name used by this auth provider defaults to "<code>
 * {@value #DEFAULT_SASL_SERVICE_NAME}</code>".
 *
 * <p><strong>Important</strong>: the SASL protocol name should match the username of the Kerberos
 * service principal used by the DSE server. This information is specified in the dse.yaml file by
 * the {@code service_principal} option under the <a
 * href="https://docs.datastax.com/en/dse/5.1/dse-admin/datastax_enterprise/config/configDseYaml.html#configDseYaml__refKerbSupport">kerberos_options</a>
 * section, and <em>may vary from one DSE installation to another</em> – especially if you installed
 * DSE with an automated package installer.
 *
 * <p>For example, if your dse.yaml file contains the following:
 *
 * <pre>{@code
 * kerberos_options:
 *     ...
 *     service_principal: cassandra/my.host.com@MY.REALM.COM
 * }</pre>
 *
 * The correct SASL protocol name to use when authenticating against this DSE server is "{@code
 * cassandra}".
 *
 * <p>Should you need to change the SASL protocol name, use one of the methods below:
 *
 * <ol>
 *   <li>Specify the service name in the driver config.
 *       <pre>
 * dse-java-driver {
 *   auth-provider {
 *     class = com.datastax.dse.driver.internal.core.auth.DseGssApiAuthProvider
 *     service = "alternate"
 *   }
 * }
 * </pre>
 *   <li>Specify the service name with the {@code dse.sasl.service} system property when starting
 *       your application, e.g. {@code -Ddse.sasl.service=cassandra}.
 * </ol>
 *
 * If a non-null SASL service name is provided to the aforementioned config, that name takes
 * precedence over the contents of the {@code dse.sasl.service} system property.
 *
 * <p>Should internal sasl properties need to be set such as qop. This can be accomplished by
 * including a sasl-properties in the driver config, for example:
 *
 * <pre>
 * dse-java-driver {
 *   auth-provider {
 *     class = com.datastax.dse.driver.internal.core.auth.DseGssApiAuthProvider
 *     sasl-properties {
 *       javax.security.sasl.qop = "auth-conf"
 *     }
 *   }
 * }
 * </pre>
 *
 * @see <a
 *     href="http://docs.datastax.com/en/dse/5.1/dse-admin/datastax_enterprise/security/securityTOC.html">Authenticating
 *     a DSE cluster with Kerberos</a>
 */
@ThreadSafe
public class DseGssApiAuthProvider implements AuthProvider {
  /** The default SASL service name used by this auth provider. */
  public static final String DEFAULT_SASL_SERVICE_NAME = "dse";

  /** The name of the system property to use to specify the SASL service name. */
  public static final String SASL_SERVICE_NAME_PROPERTY = "dse.sasl.service";

  private final DriverExecutionProfile config;

  public DseGssApiAuthProvider(DriverContext context) {
    this.config = context.getConfig().getDefaultProfile();
  }

  @NonNull
  @Override
  public Authenticator newAuthenticator(
      @NonNull EndPoint endPoint, @NonNull String serverAuthenticator)
      throws AuthenticationException {

    // it is valid to have some of these set to null
    String authorizationId = null;
    String saslService = null;
    Map<String, String> saslPropertiesMap = new HashMap<>();
    Map<String, String> loginConfigurationMap;

    // A login configuration is always necessary, throw an exception if that option is missing.
    AuthUtils.validateConfigPresent(
        config,
        DseGssApiAuthProvider.class.getName(),
        endPoint,
        DseDriverOption.AUTH_PROVIDER_LOGIN_CONFIGURATION);
    if (config.isDefined(DseDriverOption.AUTH_PROVIDER_AUTHORIZATION_ID)) {
      authorizationId = config.getString(DseDriverOption.AUTH_PROVIDER_AUTHORIZATION_ID);
    }
    if (config.isDefined(DseDriverOption.AUTH_PROVIDER_SERVICE)) {
      saslService = config.getString(DseDriverOption.AUTH_PROVIDER_SERVICE);
    }
    if (config.isDefined(DseDriverOption.AUTH_PROVIDER_SASL_PROPERTIES)) {
      saslPropertiesMap = config.getStringMap(DseDriverOption.AUTH_PROVIDER_SASL_PROPERTIES);
    }

    loginConfigurationMap = config.getStringMap(DseDriverOption.AUTH_PROVIDER_LOGIN_CONFIGURATION);
    Configuration loginConfiguration = fetchLoginConfiguration(loginConfigurationMap);
    return new GssApiAuthenticator(
        serverAuthenticator,
        authorizationId,
        endPoint,
        loginConfiguration,
        saslService,
        saslPropertiesMap);
  }

  @Override
  public void onMissingChallenge(@NonNull EndPoint endPoint) {
    // ignore
  }

  @Override
  public void close() throws Exception {
    // nothing to do
  }

  /**
   * Creates a configuration that depends on the given keytab file for authenticating the given
   * user.
   */
  public static Configuration fetchLoginConfiguration(Map<String, String> options) {
    return new Configuration() {

      @Override
      public AppConfigurationEntry[] getAppConfigurationEntry(String name) {
        return new AppConfigurationEntry[] {
          new AppConfigurationEntry(
              "com.sun.security.auth.module.Krb5LoginModule",
              AppConfigurationEntry.LoginModuleControlFlag.REQUIRED,
              options)
        };
      }
    };
  }

  private static class GssApiAuthenticator extends BaseDseAuthenticator {
    private static final String JAAS_CONFIG_ENTRY = "DseClient";
    private static final String[] SUPPORTED_MECHANISMS = new String[] {"GSSAPI"};
    private static final Map<String, String> DEFAULT_PROPERTIES =
        ImmutableMap.<String, String>builder()
            .put(Sasl.SERVER_AUTH, "true")
            .put(Sasl.QOP, "auth")
            .build();
    private static final ByteBuffer EMPTY_BYTE_ARRAY =
        ByteBuffer.wrap(new byte[0]).asReadOnlyBuffer();
    private static final ByteBuffer MECHANISM =
        ByteBuffer.wrap("GSSAPI".getBytes(Charsets.UTF_8)).asReadOnlyBuffer();
    private static final ByteBuffer SERVER_INITIAL_CHALLENGE =
        ByteBuffer.wrap("GSSAPI-START".getBytes(Charsets.UTF_8)).asReadOnlyBuffer();

    private Subject subject;
    private SaslClient saslClient;
    private EndPoint endPoint;

    private GssApiAuthenticator(
        String authenticator,
        String authorizationId,
        EndPoint endPoint,
        Configuration loginConfiguration,
        String saslService,
        Map<String, String> saslProperties) {
      super(authenticator);
      // Apply any user saslProperties defined in config to our default set of properties
      ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
      builder.putAll(saslProperties);
      DEFAULT_PROPERTIES.forEach(
          (k, v) -> {
            if (!saslProperties.containsKey(k)) {
              builder.put(k, v);
            }
          });
      ImmutableMap<String, String> finalSaslProperties = builder.build();
      try {
        this.endPoint = endPoint;
        String protocol = saslService;
        if (protocol == null) {
          protocol = System.getProperty(SASL_SERVICE_NAME_PROPERTY, DEFAULT_SASL_SERVICE_NAME);
        }
        LoginContext login = new LoginContext(JAAS_CONFIG_ENTRY, null, null, loginConfiguration);
        login.login();
        subject = login.getSubject();
        saslClient =
            Sasl.createSaslClient(
                SUPPORTED_MECHANISMS,
                authorizationId,
                protocol,
                ((InetSocketAddress) endPoint.resolve()).getAddress().getCanonicalHostName(),
                finalSaslProperties,
                null);
      } catch (LoginException | SaslException e) {
        throw new AuthenticationException(endPoint, e.getMessage());
      }
    }

    @NonNull
    @Override
    public ByteBuffer getMechanism() {
      return MECHANISM;
    }

    @NonNull
    @Override
    public ByteBuffer getInitialServerChallenge() {
      return SERVER_INITIAL_CHALLENGE;
    }

    @Nullable
    @Override
    public ByteBuffer evaluateChallengeSync(@Nullable ByteBuffer challenge) {

      byte[] challengeBytes;
      if (SERVER_INITIAL_CHALLENGE.equals(challenge)) {
        if (!saslClient.hasInitialResponse()) {
          return EMPTY_BYTE_ARRAY;
        }
        challengeBytes = new byte[0];
      } else {
        // The native protocol spec says the incoming challenge can be null depending on the
        // implementation. But saslClient.evaluateChallenge clearly documents that the byte array
        // can't be null, which probably means that a SASL authenticator never sends back null.
        if (challenge == null) {
          throw new AuthenticationException(this.endPoint, "Unexpected null challenge from server");
        }
        challengeBytes = Bytes.getArray(challenge);
      }
      try {

        return ByteBuffer.wrap(
            Subject.doAs(
                subject,
                new PrivilegedExceptionAction<byte[]>() {
                  @Override
                  public byte[] run() throws SaslException {
                    return saslClient.evaluateChallenge(challengeBytes);
                  }
                }));
      } catch (PrivilegedActionException e) {
        throw new AuthenticationException(this.endPoint, e.getMessage(), e.getException());
      }
    }
  }
}
