/*
 * Copyright DataStax, Inc.
 *
 * This software can be used solely with DataStax Enterprise. Please consult the license at
 * http://www.datastax.com/terms/datastax-dse-driver-license-terms
 */
package com.datastax.dse.driver.internal.core.auth;

import com.datastax.dse.driver.api.core.config.DseDriverOption;
import com.datastax.oss.driver.api.core.auth.AuthProvider;
import com.datastax.oss.driver.api.core.auth.AuthenticationException;
import com.datastax.oss.driver.api.core.auth.Authenticator;
import com.datastax.oss.driver.api.core.config.DefaultDriverOption;
import com.datastax.oss.driver.api.core.config.DriverExecutionProfile;
import com.datastax.oss.driver.api.core.context.DriverContext;
import com.datastax.oss.driver.api.core.metadata.EndPoint;
import com.datastax.oss.driver.shaded.guava.common.primitives.Bytes;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import net.jcip.annotations.ThreadSafe;

/**
 * An authentication provider that supports SASL authentication using the PLAIN mechanism to connect
 * to DSE clusters secured with DseAuthenticator.
 *
 * <p>To activate this provider, an {@code auth-provider} section must be included in the driver
 * configuration, for example:
 *
 * <pre>
 * dse-java-driver {
 *   auth-provider {
 *     class = com.datastax.dse.driver.internal.core.auth.DsePlainTextAuthProvider
 *     username = user0
 *     password = mypassword
 *     authorization-id = user1
 *   }
 * }
 * </pre>
 *
 * See the {@code dse-reference.conf} file included with the driver for more information.
 */
@ThreadSafe
public class DsePlainTextAuthProvider implements AuthProvider {

  private final DriverExecutionProfile config;

  public DsePlainTextAuthProvider(DriverContext context) {
    this.config = context.getConfig().getDefaultProfile();
  }

  @NonNull
  @Override
  public Authenticator newAuthenticator(
      @NonNull EndPoint endPoint, @NonNull String serverAuthenticator)
      throws AuthenticationException {
    String authorizationId;
    if (config.isDefined(DseDriverOption.AUTH_PROVIDER_AUTHORIZATION_ID)) {
      authorizationId = config.getString(DseDriverOption.AUTH_PROVIDER_AUTHORIZATION_ID);
    } else {
      authorizationId = "";
    }
    // It's not valid to use the DsePlainTextAuthProvider without a username or password, error out
    // early here
    AuthUtils.validateConfigPresent(
        config,
        DsePlainTextAuthProvider.class.getName(),
        endPoint,
        DefaultDriverOption.AUTH_PROVIDER_USER_NAME,
        DefaultDriverOption.AUTH_PROVIDER_PASSWORD);
    return new PlainTextAuthenticator(
        serverAuthenticator,
        config.getString(DefaultDriverOption.AUTH_PROVIDER_USER_NAME),
        config.getString(DefaultDriverOption.AUTH_PROVIDER_PASSWORD),
        authorizationId,
        endPoint);
  }

  @Override
  public void onMissingChallenge(@NonNull EndPoint endPoint) {
    // ignore
  }

  @Override
  public void close() throws Exception {
    // nothing to do
  }

  private static class PlainTextAuthenticator extends BaseDseAuthenticator {

    private static final ByteBuffer MECHANISM =
        ByteBuffer.wrap("PLAIN".getBytes(StandardCharsets.UTF_8)).asReadOnlyBuffer();

    private static final ByteBuffer SERVER_INITIAL_CHALLENGE =
        ByteBuffer.wrap("PLAIN-START".getBytes(StandardCharsets.UTF_8)).asReadOnlyBuffer();

    private static final byte[] NULL = new byte[] {0};

    private final byte[] authenticationId;
    private final byte[] password;
    private final byte[] authorizationId;
    private EndPoint endPoint;

    PlainTextAuthenticator(
        String serverAuthenticator,
        String authenticationId,
        String password,
        String authorizationId,
        EndPoint endPoint) {
      super(serverAuthenticator);
      this.authenticationId = authenticationId.getBytes(StandardCharsets.UTF_8);
      this.password = password.getBytes(StandardCharsets.UTF_8);
      this.authorizationId = authorizationId.getBytes(StandardCharsets.UTF_8);
      this.endPoint = endPoint;
    }

    @NonNull
    @Override
    public ByteBuffer getMechanism() {
      return MECHANISM;
    }

    @NonNull
    @Override
    public ByteBuffer getInitialServerChallenge() {
      return SERVER_INITIAL_CHALLENGE;
    }

    @Nullable
    @Override
    public ByteBuffer evaluateChallengeSync(@Nullable ByteBuffer challenge) {
      if (SERVER_INITIAL_CHALLENGE.equals(challenge)) {
        return ByteBuffer.wrap(
            Bytes.concat(authorizationId, NULL, authenticationId, NULL, password));
      }
      throw new AuthenticationException(endPoint, "Incorrect challenge from server");
    }
  }
}
