/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.CloseFuture;
import com.datastax.driver.core.ClusterNameMismatchException;
import com.datastax.driver.core.Connection;
import com.datastax.driver.core.GuavaCompatibility;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.HostDistance;
import com.datastax.driver.core.PoolingOptions;
import com.datastax.driver.core.SessionManager;
import com.datastax.driver.core.exceptions.AuthenticationException;
import com.datastax.driver.core.exceptions.BusyPoolException;
import com.datastax.driver.core.exceptions.ConnectionException;
import com.datastax.driver.core.exceptions.UnsupportedProtocolVersionException;
import com.datastax.driver.core.utils.MoreFutures;
import io.netty.util.concurrent.EventExecutor;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import shade.com.datastax.spark.connector.google.common.annotations.VisibleForTesting;
import shade.com.datastax.spark.connector.google.common.base.Throwables;
import shade.com.datastax.spark.connector.google.common.collect.Lists;
import shade.com.datastax.spark.connector.google.common.util.concurrent.AsyncFunction;
import shade.com.datastax.spark.connector.google.common.util.concurrent.FutureCallback;
import shade.com.datastax.spark.connector.google.common.util.concurrent.Futures;
import shade.com.datastax.spark.connector.google.common.util.concurrent.ListenableFuture;
import shade.com.datastax.spark.connector.google.common.util.concurrent.SettableFuture;
import shade.com.datastax.spark.connector.google.common.util.concurrent.Uninterruptibles;

class HostConnectionPool
implements Connection.Owner {
    private static final Logger logger = LoggerFactory.getLogger(HostConnectionPool.class);
    private static final int MAX_SIMULTANEOUS_CREATION = 1;
    final Host host;
    volatile HostDistance hostDistance;
    protected final SessionManager manager;
    final List<Connection> connections;
    private final AtomicInteger open;
    final AtomicInteger totalInFlight = new AtomicInteger();
    private final AtomicInteger maxTotalInFlight = new AtomicInteger();
    @VisibleForTesting
    final Set<Connection> trash = new CopyOnWriteArraySet<Connection>();
    private final Queue<PendingBorrow> pendingBorrows = new ConcurrentLinkedQueue<PendingBorrow>();
    final AtomicInteger pendingBorrowCount = new AtomicInteger();
    private final Runnable newConnectionTask;
    private final AtomicInteger scheduledForCreation = new AtomicInteger();
    private final EventExecutor timeoutsExecutor;
    private final AtomicReference<CloseFuture> closeFuture = new AtomicReference();
    protected final AtomicReference<Phase> phase = new AtomicReference<Phase>(Phase.INITIALIZING);
    private final int minAllowedStreams;

    HostConnectionPool(Host host, HostDistance hostDistance, SessionManager manager) {
        assert (hostDistance != HostDistance.IGNORED);
        this.host = host;
        this.hostDistance = hostDistance;
        this.manager = manager;
        this.newConnectionTask = new Runnable(){

            @Override
            public void run() {
                HostConnectionPool.this.addConnectionIfUnderMaximum();
                HostConnectionPool.this.scheduledForCreation.decrementAndGet();
            }
        };
        this.connections = new CopyOnWriteArrayList<Connection>();
        this.open = new AtomicInteger();
        this.minAllowedStreams = this.options().getMaxRequestsPerConnection(hostDistance) * 3 / 4;
        this.timeoutsExecutor = manager.getCluster().manager.connectionFactory.eventLoopGroup.next();
    }

    ListenableFuture<Void> initAsync(Connection reusedConnection) {
        Executor initExecutor = this.manager.cluster.manager.configuration.getPoolingOptions().getInitializationExecutor();
        final int coreSize = this.options().getCoreConnectionsPerHost(this.hostDistance);
        final ArrayList<Connection> connections = Lists.newArrayListWithCapacity(coreSize);
        ArrayList<ListenableFuture<Void>> connectionFutures = Lists.newArrayListWithCapacity(coreSize);
        int toCreate = coreSize;
        if (reusedConnection != null && toCreate > 0 && reusedConnection.setOwner(this)) {
            --toCreate;
            connections.add(reusedConnection);
            connectionFutures.add(MoreFutures.VOID_SUCCESS);
        }
        List<Connection> newConnections = this.manager.connectionFactory().newConnections(this, toCreate);
        connections.addAll(newConnections);
        for (Connection connection : newConnections) {
            ListenableFuture<Void> connectionFuture = connection.initAsync();
            connectionFutures.add(this.handleErrors(connectionFuture, initExecutor));
        }
        ListenableFuture allConnectionsFuture = Futures.allAsList(connectionFutures);
        final SettableFuture<Void> initFuture = SettableFuture.create();
        GuavaCompatibility.INSTANCE.addCallback(allConnectionsFuture, new FutureCallback<List<Void>>(){

            @Override
            public void onSuccess(List<Void> l) {
                ListIterator it = connections.listIterator();
                while (it.hasNext()) {
                    if (!((Connection)it.next()).isClosed()) continue;
                    it.remove();
                }
                HostConnectionPool.this.connections.addAll(connections);
                HostConnectionPool.this.open.set(connections.size());
                if (HostConnectionPool.this.isClosed()) {
                    initFuture.setException(new ConnectionException(HostConnectionPool.this.host.getSocketAddress(), "Pool was closed during initialization"));
                    HostConnectionPool.this.forceClose(connections);
                } else {
                    logger.debug("Created connection pool to host {} ({} connections needed, {} successfully opened)", new Object[]{HostConnectionPool.this.host, coreSize, connections.size()});
                    HostConnectionPool.this.phase.compareAndSet(Phase.INITIALIZING, Phase.READY);
                    initFuture.set(null);
                }
            }

            @Override
            public void onFailure(Throwable t) {
                HostConnectionPool.this.phase.compareAndSet(Phase.INITIALIZING, Phase.INIT_FAILED);
                HostConnectionPool.this.forceClose(connections);
                initFuture.setException(t);
            }
        }, initExecutor);
        return initFuture;
    }

    private ListenableFuture<Void> handleErrors(ListenableFuture<Void> connectionInitFuture, Executor executor) {
        return GuavaCompatibility.INSTANCE.withFallback(connectionInitFuture, new AsyncFunction<Throwable, Void>(){

            @Override
            public ListenableFuture<Void> apply(Throwable t) throws Exception {
                Throwables.propagateIfInstanceOf(t, ClusterNameMismatchException.class);
                Throwables.propagateIfInstanceOf(t, UnsupportedProtocolVersionException.class);
                Throwables.propagateIfInstanceOf(t, AuthenticationException.class);
                Throwables.propagateIfInstanceOf(t, Error.class);
                logger.warn("Error creating connection to " + HostConnectionPool.this.host, t);
                return MoreFutures.VOID_SUCCESS;
            }
        }, executor);
    }

    private void forceClose(List<Connection> connections) {
        for (Connection connection : connections) {
            connection.closeAsync().force();
        }
    }

    private PoolingOptions options() {
        return this.manager.configuration().getPoolingOptions();
    }

    ListenableFuture<Connection> borrowConnection(long timeout, TimeUnit unit, int maxQueueSize) {
        int currentCapacity;
        int oldMax;
        int inFlight;
        Phase phase = this.phase.get();
        if (phase != Phase.READY) {
            return Futures.immediateFailedFuture(new ConnectionException(this.host.getSocketAddress(), "Pool is " + (Object)((Object)phase)));
        }
        if (this.connections.isEmpty() && this.host.convictionPolicy.canReconnectNow()) {
            int coreSize = this.options().getCoreConnectionsPerHost(this.hostDistance);
            if (coreSize == 0) {
                this.maybeSpawnNewConnection();
            } else if (this.scheduledForCreation.compareAndSet(0, coreSize)) {
                for (int i = 0; i < coreSize; ++i) {
                    this.manager.blockingExecutor().submit(this.newConnectionTask);
                }
            }
            return this.enqueue(timeout, unit, maxQueueSize);
        }
        int minInFlight = Integer.MAX_VALUE;
        Connection leastBusy = null;
        for (Connection connection : this.connections) {
            int inFlight2 = connection.inFlight.get();
            if (inFlight2 >= minInFlight) continue;
            minInFlight = inFlight2;
            leastBusy = connection;
        }
        if (leastBusy == null) {
            if (this.isClosed()) {
                return Futures.immediateFailedFuture(new ConnectionException(this.host.getSocketAddress(), "Pool is shutdown"));
            }
            return this.enqueue(timeout, unit, maxQueueSize);
        }
        do {
            if ((inFlight = leastBusy.inFlight.get()) < Math.min(leastBusy.maxAvailableStreams(), this.options().getMaxRequestsPerConnection(this.hostDistance))) continue;
            return this.enqueue(timeout, unit, maxQueueSize);
        } while (!leastBusy.inFlight.compareAndSet(inFlight, inFlight + 1));
        int totalInFlightCount = this.totalInFlight.incrementAndGet();
        while (totalInFlightCount > (oldMax = this.maxTotalInFlight.get()) && !this.maxTotalInFlight.compareAndSet(oldMax, totalInFlightCount)) {
        }
        int connectionCount = this.open.get() + this.scheduledForCreation.get();
        if (connectionCount < this.options().getCoreConnectionsPerHost(this.hostDistance)) {
            this.maybeSpawnNewConnection();
        } else if (connectionCount < this.options().getMaxConnectionsPerHost(this.hostDistance) && totalInFlightCount > (currentCapacity = (connectionCount - 1) * this.options().getMaxRequestsPerConnection(this.hostDistance) + this.options().getNewConnectionThreshold(this.hostDistance))) {
            this.maybeSpawnNewConnection();
        }
        return leastBusy.setKeyspaceAsync(this.manager.poolsState.keyspace);
    }

    private ListenableFuture<Connection> enqueue(long timeout, TimeUnit unit, int maxQueueSize) {
        int count;
        if (timeout == 0L || maxQueueSize == 0) {
            return Futures.immediateFailedFuture(new BusyPoolException(this.host.getSocketAddress(), 0));
        }
        do {
            if ((count = this.pendingBorrowCount.get()) < maxQueueSize) continue;
            return Futures.immediateFailedFuture(new BusyPoolException(this.host.getSocketAddress(), maxQueueSize));
        } while (!this.pendingBorrowCount.compareAndSet(count, count + 1));
        PendingBorrow pendingBorrow = new PendingBorrow(timeout, unit, this.timeoutsExecutor);
        this.pendingBorrows.add(pendingBorrow);
        if (this.phase.get() == Phase.CLOSING) {
            pendingBorrow.setException(new ConnectionException(this.host.getSocketAddress(), "Pool is shutdown"));
        }
        return pendingBorrow.future;
    }

    void returnConnection(Connection connection, boolean busy) {
        connection.inFlight.decrementAndGet();
        this.totalInFlight.decrementAndGet();
        if (this.isClosed()) {
            this.close(connection);
            return;
        }
        if (connection.isDefunct()) {
            return;
        }
        if (connection.state.get() != Connection.State.TRASHED) {
            if (connection.maxAvailableStreams() < this.minAllowedStreams) {
                this.replaceConnection(connection);
            } else if (!busy) {
                this.dequeue(connection);
            }
        }
    }

    private void dequeue(final Connection connection) {
        while (!this.pendingBorrows.isEmpty()) {
            int inFlight;
            do {
                if ((inFlight = connection.inFlight.get()) < Math.min(connection.maxAvailableStreams(), this.options().getMaxRequestsPerConnection(this.hostDistance))) continue;
                return;
            } while (!connection.inFlight.compareAndSet(inFlight, inFlight + 1));
            final PendingBorrow pendingBorrow = this.pendingBorrows.poll();
            if (pendingBorrow == null) {
                connection.inFlight.decrementAndGet();
                continue;
            }
            this.pendingBorrowCount.decrementAndGet();
            ListenableFuture<Connection> setKeyspaceFuture = connection.setKeyspaceAsync(this.manager.poolsState.keyspace);
            if (setKeyspaceFuture.isDone()) {
                try {
                    if (pendingBorrow.set(Uninterruptibles.getUninterruptibly(setKeyspaceFuture))) {
                        this.totalInFlight.incrementAndGet();
                        continue;
                    }
                    connection.inFlight.decrementAndGet();
                }
                catch (ExecutionException e) {
                    pendingBorrow.setException(e.getCause());
                    connection.inFlight.decrementAndGet();
                }
                continue;
            }
            GuavaCompatibility.INSTANCE.addCallback(setKeyspaceFuture, new FutureCallback<Connection>(){

                @Override
                public void onSuccess(Connection c) {
                    if (pendingBorrow.set(c)) {
                        HostConnectionPool.this.totalInFlight.incrementAndGet();
                    } else {
                        connection.inFlight.decrementAndGet();
                    }
                }

                @Override
                public void onFailure(Throwable t) {
                    pendingBorrow.setException(t);
                    connection.inFlight.decrementAndGet();
                }
            });
        }
    }

    private void replaceConnection(Connection connection) {
        if (!connection.state.compareAndSet(Connection.State.OPEN, Connection.State.TRASHED)) {
            return;
        }
        this.open.decrementAndGet();
        this.maybeSpawnNewConnection();
        connection.maxIdleTime = Long.MIN_VALUE;
        this.doTrashConnection(connection);
    }

    private boolean trashConnection(Connection connection) {
        int opened;
        if (!connection.state.compareAndSet(Connection.State.OPEN, Connection.State.TRASHED)) {
            return true;
        }
        do {
            if ((opened = this.open.get()) > this.options().getCoreConnectionsPerHost(this.hostDistance)) continue;
            connection.state.set(Connection.State.OPEN);
            return false;
        } while (!this.open.compareAndSet(opened, opened - 1));
        logger.trace("Trashing {}", (Object)connection);
        connection.maxIdleTime = System.currentTimeMillis() + (long)(this.options().getIdleTimeoutSeconds() * 1000);
        this.doTrashConnection(connection);
        return true;
    }

    private void doTrashConnection(Connection connection) {
        this.connections.remove(connection);
        this.trash.add(connection);
    }

    private boolean addConnectionIfUnderMaximum() {
        int opened;
        do {
            if ((opened = this.open.get()) < this.options().getMaxConnectionsPerHost(this.hostDistance)) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened + 1));
        if (this.phase.get() != Phase.READY) {
            this.open.decrementAndGet();
            return false;
        }
        try {
            Connection newConnection = this.tryResurrectFromTrash();
            if (newConnection == null) {
                if (!this.host.convictionPolicy.canReconnectNow()) {
                    this.open.decrementAndGet();
                    return false;
                }
                logger.debug("Creating new connection on busy pool to {}", (Object)this.host);
                newConnection = this.manager.connectionFactory().open(this);
                newConnection.setKeyspace(this.manager.poolsState.keyspace);
            }
            this.connections.add(newConnection);
            newConnection.state.compareAndSet(Connection.State.RESURRECTING, Connection.State.OPEN);
            if (this.isClosed() && !newConnection.isClosed()) {
                this.close(newConnection);
                this.open.decrementAndGet();
                return false;
            }
            this.dequeue(newConnection);
            return true;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.open.decrementAndGet();
            return false;
        }
        catch (ConnectionException e) {
            this.open.decrementAndGet();
            logger.debug("Connection error to {} while creating additional connection", (Object)this.host);
            return false;
        }
        catch (AuthenticationException e) {
            this.open.decrementAndGet();
            logger.error("Authentication error while creating additional connection (error is: {})", (Object)e.getMessage());
            return false;
        }
        catch (UnsupportedProtocolVersionException e) {
            this.open.decrementAndGet();
            logger.error("UnsupportedProtocolVersionException error while creating additional connection (error is: {})", (Object)e.getMessage());
            return false;
        }
        catch (ClusterNameMismatchException e) {
            this.open.decrementAndGet();
            logger.error("ClusterNameMismatchException error while creating additional connection (error is: {})", (Object)e.getMessage());
            return false;
        }
    }

    private Connection tryResurrectFromTrash() {
        long highestMaxIdleTime = System.currentTimeMillis();
        Connection chosen = null;
        do {
            for (Connection connection : this.trash) {
                if (connection.maxIdleTime <= highestMaxIdleTime || connection.maxAvailableStreams() <= this.minAllowedStreams) continue;
                chosen = connection;
                highestMaxIdleTime = connection.maxIdleTime;
            }
            if (chosen != null) continue;
            return null;
        } while (!chosen.state.compareAndSet(Connection.State.TRASHED, Connection.State.RESURRECTING));
        logger.trace("Resurrecting {}", chosen);
        this.trash.remove(chosen);
        return chosen;
    }

    private void maybeSpawnNewConnection() {
        int inCreation;
        if (this.isClosed() || !this.host.convictionPolicy.canReconnectNow()) {
            return;
        }
        do {
            if ((inCreation = this.scheduledForCreation.get()) < 1) continue;
            return;
        } while (!this.scheduledForCreation.compareAndSet(inCreation, inCreation + 1));
        this.manager.blockingExecutor().submit(this.newConnectionTask);
    }

    @Override
    public void onConnectionDefunct(Connection connection) {
        if (connection.state.compareAndSet(Connection.State.OPEN, Connection.State.GONE)) {
            this.open.decrementAndGet();
        }
        this.connections.remove(connection);
    }

    void cleanupIdleConnections(long now) {
        if (this.isClosed()) {
            return;
        }
        this.shrinkIfBelowCapacity();
        this.cleanupTrash(now);
    }

    private void shrinkIfBelowCapacity() {
        int currentLoad = this.maxTotalInFlight.getAndSet(this.totalInFlight.get());
        int maxRequestsPerConnection = this.options().getMaxRequestsPerConnection(this.hostDistance);
        int needed = currentLoad / maxRequestsPerConnection + 1;
        if (currentLoad % maxRequestsPerConnection > this.options().getNewConnectionThreshold(this.hostDistance)) {
            ++needed;
        }
        needed = Math.max(needed, this.options().getCoreConnectionsPerHost(this.hostDistance));
        int actual = this.open.get();
        int toTrash = Math.max(0, actual - needed);
        logger.trace("Current inFlight = {}, {} connections needed, {} connections available, trashing {}", new Object[]{currentLoad, needed, actual, toTrash});
        if (toTrash <= 0) {
            return;
        }
        for (Connection connection : this.connections) {
            if (!this.trashConnection(connection) || --toTrash != 0) continue;
            return;
        }
    }

    private void cleanupTrash(long now) {
        for (Connection connection : this.trash) {
            if (connection.maxIdleTime >= now || !connection.state.compareAndSet(Connection.State.TRASHED, Connection.State.GONE)) continue;
            if (connection.inFlight.get() == 0) {
                logger.trace("Cleaning up {}", (Object)connection);
                this.trash.remove(connection);
                this.close(connection);
                continue;
            }
            connection.state.set(Connection.State.TRASHED);
        }
    }

    private void close(Connection connection) {
        connection.closeAsync();
    }

    final boolean isClosed() {
        return this.closeFuture.get() != null;
    }

    final CloseFuture closeAsync() {
        CloseFuture future = this.closeFuture.get();
        if (future != null) {
            return future;
        }
        this.phase.set(Phase.CLOSING);
        for (PendingBorrow pendingBorrow : this.pendingBorrows) {
            pendingBorrow.setException(new ConnectionException(this.host.getSocketAddress(), "Pool is shutdown"));
        }
        future = new CloseFuture.Forwarding(this.discardAvailableConnections());
        return this.closeFuture.compareAndSet(null, future) ? future : this.closeFuture.get();
    }

    int opened() {
        return this.open.get();
    }

    int trashed() {
        return this.trash.size();
    }

    private List<CloseFuture> discardAvailableConnections() {
        ArrayList<CloseFuture> futures = new ArrayList<CloseFuture>(this.connections.size() + this.trash.size());
        for (final Connection connection : this.connections) {
            CloseFuture future = connection.closeAsync();
            future.addListener(new Runnable(){

                @Override
                public void run() {
                    if (connection.state.compareAndSet(Connection.State.OPEN, Connection.State.GONE)) {
                        HostConnectionPool.this.open.decrementAndGet();
                    }
                }
            }, GuavaCompatibility.INSTANCE.sameThreadExecutor());
            futures.add(future);
        }
        for (final Connection connection : this.trash) {
            futures.add(connection.closeAsync());
        }
        return futures;
    }

    void ensureCoreConnections() {
        int opened;
        if (this.isClosed()) {
            return;
        }
        if (!this.host.convictionPolicy.canReconnectNow()) {
            return;
        }
        for (int i = opened = this.open.get(); i < this.options().getCoreConnectionsPerHost(this.hostDistance); ++i) {
            this.scheduledForCreation.incrementAndGet();
            this.manager.blockingExecutor().submit(this.newConnectionTask);
        }
    }

    private class PendingBorrow {
        final SettableFuture<Connection> future = SettableFuture.create();
        final Future<?> timeoutTask;

        PendingBorrow(final long timeout, final TimeUnit unit, EventExecutor timeoutsExecutor) {
            this.timeoutTask = timeoutsExecutor.schedule(new Runnable(){

                @Override
                public void run() {
                    PendingBorrow.this.future.setException(new BusyPoolException(HostConnectionPool.this.host.getSocketAddress(), timeout, unit));
                }
            }, timeout, unit);
        }

        boolean set(Connection connection) {
            boolean succeeded = this.future.set(connection);
            this.timeoutTask.cancel(false);
            return succeeded;
        }

        void setException(Throwable exception) {
            this.future.setException(exception);
            this.timeoutTask.cancel(false);
        }
    }

    static class PoolState {
        volatile String keyspace;

        PoolState() {
        }

        void setKeyspace(String keyspace) {
            this.keyspace = keyspace;
        }
    }

    private static enum Phase {
        INITIALIZING,
        READY,
        INIT_FAILED,
        CLOSING;

    }
}

