package com.dooapp.gaedo.finders.informers;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import com.dooapp.gaedo.finders.FieldInformer;
import com.dooapp.gaedo.finders.FieldInformerAPI;
import com.dooapp.gaedo.finders.QueryExpression;
import com.dooapp.gaedo.finders.expressions.AnythingExpression;
import com.dooapp.gaedo.finders.expressions.EqualsExpression;
import com.dooapp.gaedo.finders.expressions.Expressions;
import com.dooapp.gaedo.properties.Property;

public class ObjectFieldInformer implements
		FieldInformerAPI {

	/**
	 * Field associated to expressions generated by this informer
	 */
	protected final Property source;
	
	protected Collection<Property> parentPath = Collections.emptyList();

	public ObjectFieldInformer(Property source) {
		super();
		this.source = source;
	}

	/**
	 * The most common check : is field value equals to input value
	 * 
	 * @param value
	 *            expected value
	 * @return a {@link EqualsExpression} checking field value is given value
	 */
	public QueryExpression equalsTo(Object value) {
		return new EqualsExpression(source, getFieldPath(), value);
	}

	/**
	 * The second most common check : is field value different from input value
	 * 
	 * @param value
	 *            expected value
	 * @return a {@link EqualsExpression} checking field value is given value
	 *         embedded in {@link Expressions#not(QueryExpression)} call
	 */
	public QueryExpression differentFrom(Object value) {
		return Expressions.not(equalsTo(value));
	}

	/**
	 * The thirsd most common check : this value is anything. It's a common way to have the field have any value.
	 * @return
	 */
	public QueryExpression isAnything() {
		return new AnythingExpression(source, getFieldPath());
	}

	@Override
	public String toString() {
		StringBuilder sOut = new StringBuilder();
		sOut.append(getClass().getName()).append(" on field ").append(source.toGenericString());
		return sOut.toString();
	}

	@Override
	public Property getField() {
		return source;
	}

	/**
	 * @return
	 * @throws CloneNotSupportedException
	 * @see java.lang.Object#clone()
	 */
	@Override
	protected ObjectFieldInformer clone() {
		return new ObjectFieldInformer(source);
	}

	@Override
	public Iterable<Property> getFieldPath() {
		List<Property> returned  =new LinkedList<Property>(parentPath);
		returned.add(source);
		return returned;
	}

	@Override
	public FieldInformer with(Collection<Property> propertyPath) {
		ObjectFieldInformer returned = clone();
		returned.parentPath = propertyPath;
		return returned;
	}

	/**
	 * @return
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((source == null) ? 0 : source.hashCode());
		return result;
	}

	/**
	 * @param obj
	 * @return
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ObjectFieldInformer other = (ObjectFieldInformer) obj;
		if (source == null) {
			if (other.source != null)
				return false;
		} else if (!source.equals(other.source))
			return false;
		return true;
	}
}
