package com.doopp.common.util;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.*;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeader;
import org.apache.http.ssl.SSLContexts;
import org.apache.http.ssl.TrustStrategy;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.SSLContext;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

public class HttpClient {

    private static final Logger log = LoggerFactory.getLogger(HttpClient.class);

    private static CloseableHttpClient closeableHttpClient;

    private final static Map<String, String> baseHeaders = new HashMap<String, String>(){{
        put("connection", "keep-alive");
        put("user-agent", "Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/96.0.4664.45 Safari/537.36");
        put("accept", "text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.9");
        put("accept-encoding", "gzip, deflate, br");
        put("accept-language", "zh-CN,zh;q=0.9,en;q=0.8");
        put("cache-control", "max-age=0");
    }};

    private final static Map<String, String> jsonHeaders = new HashMap<String, String>(){{
        put("content-type", "application/json");
    }};

    public static void setDefaultCloseableHttpClient() {
        HttpClient.closeableHttpClient = getDefaultCloseableHttpClient();
    }

    public static void setCloseableHttpClient(CloseableHttpClient closeableHttpClient) {
        HttpClient.closeableHttpClient = closeableHttpClient;
    }

    public static void setCloseableHttpClient(Integer maxTotal, Integer maxPerRoute, Integer socketTimeOut, Integer connectTimeout, Integer requestTimeout, Integer idleConnections) {
        HttpClient.closeableHttpClient = getCloseableHttpClient(maxTotal, maxPerRoute, socketTimeOut, connectTimeout, requestTimeout, idleConnections);
    }

    public static <T> T jsonGet(String url, Map<String, String> headerMap, TypeReference<T> type) {
        HttpGet httpGet = new HttpGet(url);
        httpGet.setHeaders(headers(headerMap, jsonHeaders));
        return jsonRequest(httpGet, type);
    }

    public static <T> T jsonPost(String url, Map<String, String> headerMap, Object obj, TypeReference<T> type) {
        HttpPost httpPost = new HttpPost(url);
        httpPost.setHeaders(headers(headerMap, jsonHeaders));
        httpPost.setEntity(object2JsonEntity(obj));
        return jsonRequest(httpPost, type);
    }

    public static <T> T jsonPut(String url, Map<String, String> headerMap, Object obj, TypeReference<T> type) {
        HttpPut httpPut = new HttpPut(url);
        httpPut.setHeaders(headers(headerMap, jsonHeaders));
        httpPut.setEntity(object2JsonEntity(obj));
        return jsonRequest(httpPut, type);
    }

    public static <T> T jsonGet(String url, Map<String, String> headerMap, Class<T> clazz) {
        HttpGet httpGet = new HttpGet(url);
        httpGet.setHeaders(headers(headerMap, jsonHeaders));
        return jsonRequest(httpGet, clazz);
    }

    public static <T> T jsonPost(String url, Map<String, String> headerMap, Object obj, Class<T> clazz) {
        HttpPost httpPost = new HttpPost(url);
        httpPost.setHeaders(headers(headerMap, jsonHeaders));
        httpPost.setEntity(object2JsonEntity(obj));
        return jsonRequest(httpPost, clazz);
    }

    public static <T> T jsonPut(String url, Map<String, String> headerMap, Object obj, Class<T> clazz) {
        HttpPut httpPut = new HttpPut(url);
        httpPut.setHeaders(headers(headerMap, jsonHeaders));
        httpPut.setEntity(object2JsonEntity(obj));
        return jsonRequest(httpPut, clazz);
    }

    private static HttpEntity object2JsonEntity(Object object) {
        String json = "{}";
        try {
            json = ApplicationContextUtil.getBean(ObjectMapper.class).writeValueAsString(object);
        }
        catch (Exception e) {
            log.info("object2Json exception : {}", object);
        }
        return new StringEntity(json, "UTF-8");
    }

    public static <T> T jsonRequest(HttpRequestBase httpRequest, TypeReference<T> typeReference) {
        String result = stringResult(httpRequest);
        try {
            return ApplicationContextUtil.getBean(ObjectMapper.class).readValue(result, typeReference);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> T jsonRequest(HttpRequestBase httpRequest, Class<T> clazz) {
        String result = stringResult(httpRequest);
        try {
            return ApplicationContextUtil.getBean(ObjectMapper.class).readValue(result, clazz);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static String get(String url) {
        HttpGet httpGet = new HttpGet(url);
        httpGet.setHeaders(headers(new HashMap<>()));
        return stringResult(httpGet);
    }

    public static String get(String url, Map<String, String> headerMap) {
        HttpGet httpGet = new HttpGet(url);
        httpGet.setHeaders(headers(headerMap));
        return stringResult(httpGet);
    }

    public static String post(String url) {
        HttpPost httpRequest = new HttpPost(url);
        httpRequest.setHeaders(headers(new HashMap<>()));
        httpRequest.setEntity(new ByteArrayEntity(new byte[0]));
        return stringResult(httpRequest);
    }

    public static String post(String url, Map<String, String> headerMap) {
        HttpPost httpRequest = new HttpPost(url);
        httpRequest.setHeaders(headers(headerMap));
        httpRequest.setEntity(new ByteArrayEntity(new byte[0]));
        return stringResult(httpRequest);
    }

    public static String post(String url, Map<String, String> headerMap, HttpEntity httpEntity) {
        HttpPost httpRequest = new HttpPost(url);
        httpRequest.setHeaders(headers(headerMap));
        httpRequest.setEntity(httpEntity);
        return stringResult(httpRequest);
    }

    public static String put(String url, Map<String, String> headerMap, HttpEntity httpEntity) {
        HttpPut httpRequest = new HttpPut(url);
        httpRequest.setHeaders(headers(headerMap));
        httpRequest.setEntity(httpEntity);
        return stringResult(httpRequest);
    }

    public static byte[] bytesGet(String url, Map<String, String> headerMap) {
        HttpGet httpRequest = new HttpGet(url);
        httpRequest.setHeaders(headers(headerMap));
        return bytesResult(httpRequest);
    }

    public static byte[] bytesPost(String url, Map<String, String> headerMap, HttpEntity httpEntity) {
        HttpPost httpRequest = new HttpPost(url);
        httpRequest.setHeaders(headers(headerMap));
        httpRequest.setEntity(httpEntity);
        return bytesResult(httpRequest);
    }

    public static byte[] bytesPut(String url, Map<String, String> headerMap, HttpEntity httpEntity) {
        HttpPut httpRequest = new HttpPut(url);
        httpRequest.setHeaders(headers(headerMap));
        httpRequest.setEntity(httpEntity);
        return bytesResult(httpRequest);
    }

    private static String stringResult(HttpRequestBase httpRequest) {
        return baseExecute(httpRequest, (httpEntity)->{
            try {
                return EntityUtils.toString(httpEntity, StandardCharsets.UTF_8);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }

    private static byte[] bytesResult(HttpRequestBase httpRequest) {
        return baseExecute(httpRequest, (httpEntity)->{
            try {
                return EntityUtils.toByteArray(httpEntity);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }

    private static <T> T baseExecute(HttpRequestBase request, Function<HttpEntity, T> callback) {
        if (CommonUtil.isEmpty(closeableHttpClient)) {
            throw new RuntimeException("HttpClient can not created");
        }
        try (CloseableHttpResponse response = closeableHttpClient.execute(request)){
            return callback.apply(response.getEntity());
            // return EntityUtils.toByteArray(response.getEntity());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static CloseableHttpClient getDefaultCloseableHttpClient() {
        return getCloseableHttpClient(50,
                20,
                60000,
                60000,
                60000,
                60
        );
    }



    private static CloseableHttpClient getCloseableHttpClient(Integer maxTotal,
                                                              Integer maxPerRoute,
                                                              Integer socketTimeOut,
                                                              Integer connectTimeout,
                                                              Integer requestTimeout,
                                                              Integer idleConnections) {
        // https config
        TrustStrategy acceptingTrustStrategy = (X509Certificate[] chain, String authType) -> true;
        SSLContext sslContext;
        try {
            sslContext = SSLContexts.custom()
                    .loadTrustMaterial(null, acceptingTrustStrategy)
                    .build();
        }
        catch (NoSuchAlgorithmException | KeyStoreException | KeyManagementException e) {
            throw new RuntimeException(e);
        }

        SSLConnectionSocketFactory csf = new SSLConnectionSocketFactory(sslContext
                ,null, null, NoopHostnameVerifier.INSTANCE);

        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", PlainConnectionSocketFactory.getSocketFactory())
                .register("https", csf)
                .build();

        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(registry);
        // 最大连接数 50
        connectionManager.setMaxTotal(maxTotal);
        // 路由链接数 20
        connectionManager.setDefaultMaxPerRoute(maxPerRoute);
        RequestConfig requestConfig = RequestConfig.custom()
                .setSocketTimeout(socketTimeOut)
                .setConnectTimeout(connectTimeout)
                .setConnectionRequestTimeout(requestTimeout)
                .setRedirectsEnabled(true)
                .setCircularRedirectsAllowed(true)
                .build();

        return HttpClients.custom().setDefaultRequestConfig(requestConfig)
                .setConnectionManager(connectionManager)
                .evictExpiredConnections()
                .evictIdleConnections(idleConnections, TimeUnit.SECONDS)
                .build();
    }

    @SafeVarargs
    private static Header[] headers(Map<String, String>... moreHeaderMaps) {
        Map<String, String> headerMap = new HashMap<>(baseHeaders);
        if (moreHeaderMaps.length>=1) {
            for (Map<String, String> moreHeaderMap : moreHeaderMaps) {
                if (!CommonUtil.isEmpty(moreHeaderMap)) {
                    headerMap.putAll(moreHeaderMap);
                }
            }
        }
        Header[] headers = new Header[headerMap.size()];
        int ii=0;
        for (String key : headerMap.keySet()) {
            headers[ii] = new BasicHeader(key, headerMap.get(key));
            ii++;
        }
        return headers;
    }
}
