package com.doopp.common.util;

import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.serializer.JdkSerializationRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.util.HashSet;
import java.util.Set;
import java.util.function.Function;

public class LettuceClient {

    private final LettuceConnectionFactory lettuceConnectionFactory;

    private final RedisSerializer<String> stringSerializer = new StringRedisSerializer();

    private final RedisSerializer<Object> objectSerializer = new JdkSerializationRedisSerializer();

    public LettuceClient(LettuceConnectionFactory lettuceConnectionFactory) {
        this.lettuceConnectionFactory = lettuceConnectionFactory;
    }

    private byte[] sb(String str) {
        return str==null ? null : stringSerializer.serialize(str);
    }

    private String bs(byte[] bytes) {
        return bytes==null ? null : stringSerializer.deserialize(bytes);
    }

    public String get(String key) {
        byte[] bytes = this.executeLettuce(
                (redisConnection)-> redisConnection.get(sb(key))
        );
        return bs(bytes);
    }

    public void set(String key, String value) {
        this.executeLettuce(
                (redisConnection)-> redisConnection.set(sb(key), sb(value))
        );
    }

    public void set(byte[] key, Object object) {
        this.executeLettuce((redisConnection)->{
            byte[] objectBytes = objectSerializer.serialize(object);
            if (objectBytes!=null) {
                redisConnection.set(key, objectBytes);
            }
            return key;
        });
    }

    public void setEx(String key, long seconds, String value) {
        this.executeLettuce((redisConnection)->{
            redisConnection.setEx(sb(key), seconds, sb(value));
            return key;
        });
    }

    public void setEx(byte[] key, long seconds, Object obj) {
        this.executeLettuce((redisConnection)->{
            byte[] objectBytes = objectSerializer.serialize(obj);
            if (objectBytes!=null) {
                redisConnection.setEx(key, seconds, objectBytes);
            }
            return key;
        });
    }

    public <T> T get(byte[] key, Class<T> clazz) {
        return this.executeLettuce((redisConnection)->{
            byte[] res = redisConnection.get(key);
            if (res==null) {
                return null;
            }
            Object obj = objectSerializer.deserialize(res);
            return obj==null ? null : clazz.cast(obj);
        });
    }

    public void del(String... keys) {
        Set<byte[]> buffer=new HashSet<>();
        for (String key : keys) {
            buffer.add(key.getBytes());
        }
        del(buffer.toArray(new byte[][]{}));
    }

    public void del(byte[]... keys) {
        this.executeLettuce((shardedJedis)->{
            for (byte[] key : keys) {
                shardedJedis.del(key);
            }
            return null;
        });
    }

    public void hSet(String key, String field, String value) {
        this.executeLettuce(
                (redisConnection)-> redisConnection.hSet(sb(key), sb(field), sb(value))
        );
    }

    public String hGet(String key, String field) {
        byte[] bytes = this.executeLettuce(
                (redisConnection)-> redisConnection.hGet(sb(key), sb(field))
        );
        return bs(bytes);
    }

    public void hSet(byte[] key, byte[] field, Object object) {
        this.executeLettuce((redisConnection)->{
            byte[] objectBytes = objectSerializer.serialize(object);
            if (objectBytes!=null) {
                redisConnection.hSet(key, field, objectBytes);
            }
            return key;
        });
    }

    public <T> T hGet(byte[] key, byte[] field, Class<T> clazz) {
        return this.executeLettuce((redisConnection)->{
            byte[] res = redisConnection.hGet(key, field);
            if (res==null) {
                return null;
            }
            Object obj = objectSerializer.deserialize(res);
            return obj==null ? null : clazz.cast(obj);
        });
    }

    private <T> T executeLettuce(Function<RedisConnection, T> function) {
        try (RedisConnection redisConnection = this.lettuceConnectionFactory.getConnection()) {
            return function.apply(redisConnection);
        }
        catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }
}
