package com.doopp.common.util;

import com.doopp.common.FileUploadParams;

import javax.activation.MimetypesFileTypeMap;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;


public class StorageUtils {



    private Object _client;

    public void configObs(String endPoint, String bucketName, String accessKeyId, String accessKeySecret) {
        _client = new OBSClient(endPoint, bucketName, accessKeyId, accessKeySecret);
    }

    public void configOss(String endPoint, String bucketName, String accessKeyId, String accessKeySecret) {
        _client = new OSSClient(endPoint, bucketName, accessKeyId, accessKeySecret);
    }

    public void configS3(String endPoint, String bucket, String region, String accessKey, String accessSecret) {
        _client = new S3Client(endPoint, bucket, region, accessKey, accessSecret);
    }

    public FileUploadParams publicPutParams(String fileName, String prefix) {
        if (_client instanceof S3Client) {
            return ((S3Client) _client).publicPutParams(fileName, prefix);
        }
        else if (_client instanceof OBSClient) {
            return ((OBSClient) _client).publicPutParams(fileName, prefix);
        }
        else if (_client instanceof OSSClient) {
            return ((OSSClient) _client).publicPutParams(fileName, prefix);
        }
        return null;
    }
}

class BaseClient {

    protected final String bucket;

    protected final String endPoint;

    protected final String accessKey;

    protected final String accessSecret;

    protected String ACL_HEADER_KEY;

    protected String AZT_LABEL;

    public BaseClient(String endPoint, String bucket, String accessKey, String accessSecret) {
        this.endPoint = endPoint;
        this.accessKey = accessKey;
        this.accessSecret = accessSecret;
        this.bucket = bucket;
    }

    public FileUploadParams privatePutParams(String fileName, String prefix) {
        String contentType = MimetypesFileTypeMap.getDefaultFileTypeMap().getContentType(fileName);
        return uploadParams("private", prefix, fileName, contentType);
    }

    public FileUploadParams publicPutParams(String fileName, String prefix) {
        String contentType = fileName.endsWith(".png")
                ? "image/png"
                : MimetypesFileTypeMap.getDefaultFileTypeMap().getContentType(fileName);
        return uploadParams("public-read", prefix, fileName, contentType);
    }

    private FileUploadParams uploadParams(String acl, String prefix, String fileName, String contentType) {
        // signature
        String uri = this.buildSourceUri(prefix, fileName);
        String date = this.getDate();
        String signature = putSignature(acl, contentType, date, uri);
        // map
        Map<String, String> headerMap = new HashMap<>();
        headerMap.put("Host", this.getHost());
        headerMap.put("Date", date);
        headerMap.put("Authorization", this.getAuthorization(signature));
        headerMap.put(ACL_HEADER_KEY, acl);
        headerMap.put("Content-Type", contentType);
        // build params
        FileUploadParams params = new FileUploadParams();
        params.setHeaders(headerMap);
        params.setUri(uri);
        params.setUrl(this.getUrl(uri));
        params.setMethod("PUT");
        // log.info("params >>> {}", params);
        return params;
    }

    protected String putSignature(String acl, String contentType, String dateValue, String resourceUri) {
        String target = "PUT"
                + "\n"
                + "\n" + contentType
                + "\n" + dateValue
                + "\n" + ACL_HEADER_KEY + ":" + acl
                + "\n" + "/" + this.bucket + resourceUri;
        return sign(accessSecret, target);
    }

    protected String getHost() {
        String endPoint = this.endPoint.startsWith("http")
                ? this.endPoint.substring(this.endPoint.indexOf("://") + 3)
                : this.endPoint;
        return this.bucket + "." + endPoint;
    }

    protected String getUrl(String uri) {
        return "https://" + this.getHost() + uri;
    }

    protected String getAuthorization(String signature) {
        return AZT_LABEL + " " + accessKey + ":" + signature;
    }

    protected String buildSourceUri(String prefix, String fileName) {
        prefix = prefix.startsWith("/") ? prefix.substring(1) : prefix;
        // 如果含有文件类型，prefix 就是完整文件路径
        if (prefix.contains(".")) {
            return "/" + prefix;
        }
        String suffix = fileName.substring(fileName.lastIndexOf(".")).toLowerCase();
        String[] dateSplit = new SimpleDateFormat("yyyy-MM-dd").format(new Date()).split("-");
        return "/" + prefix +
                "/" + dateSplit[0] +
                "/" + dateSplit[1] +
                "/" + UUID.randomUUID() + suffix;
    }

    protected String getDate() {
        // SimpleTimeZone tz = new SimpleTimeZone(0, "Out Timezone");
        // TimeZone.setDefault(tz);
        // return new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss", Locale.ENGLISH).format(new Date()) + " GMT";
        DateFormat dateFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss", Locale.ENGLISH);
        SimpleTimeZone tz = new SimpleTimeZone(0, "Out Timezone");
        dateFormat.setTimeZone(tz);
        return dateFormat.format(new Date()) + " GMT";
    }

    private static final String ALGORITHM_NAME = "HmacSHA1";

    private static final String ENCODING = "UTF-8";

    private String sign(String accessSecret, String target) {
        try {
            Mac mac = Mac.getInstance(ALGORITHM_NAME);
            mac.init(new SecretKeySpec(accessSecret.getBytes(ENCODING), ALGORITHM_NAME));
            byte[] signData = mac.doFinal(target.getBytes(ENCODING));
            return Base64.getEncoder().encodeToString(signData);
        }
        catch (NoSuchAlgorithmException | UnsupportedEncodingException | InvalidKeyException e) {
            throw new IllegalArgumentException(e.toString());
        }
    }
}

class OSSClient extends BaseClient {

    public OSSClient(String endPoint, String bucket, String accessKey, String accessSecret) {
        super(endPoint, bucket, accessKey, accessSecret);
        this.ACL_HEADER_KEY = "x-oss-object-acl";
        this.AZT_LABEL = "OSS";
    }
}

class OBSClient extends BaseClient {

    public OBSClient(String endPoint, String bucket, String accessKey, String accessSecret) {
        super(endPoint, bucket, accessKey, accessSecret);
        this.ACL_HEADER_KEY = "x-obs-acl";
        this.AZT_LABEL = "OBS";
    }
}

class S3Client {

    private final static String HMACAlgorithm = "AWS4-HMAC-SHA256";
    private final static String Aws4Request = "aws4_request";
    private final static String ServiceName = "s3";
    private final static String AclHeaderName = "x-amz-acl";
    private final static String AztLabel = "AWS";
    private final static String StorageClass= "REDUCED_REDUNDANCY";
    private final static String AuthorizationTemp = "AWS4-HMAC-SHA256 Credential=%s/%s/%s/s3/aws4_request,SignedHeaders=date;host;x-amz-content-sha256;x-amz-date;x-amz-storage-class,Signature=%s";

    private final String endPoint;
    private final String bucket;
    private final String region;
    private final String accessKey;
    private final String accessSecret;

    public S3Client(String endPoint, String bucket, String region, String accessKey, String accessSecret) {
        this.endPoint = endPoint;
        this.bucket = bucket;
        this.region = region;
        this.accessKey = accessKey;
        this.accessSecret = accessSecret;
    }

    public FileUploadParams publicPutParams(String fileName, String prefix) {
        String contentType = fileName.endsWith(".png")
                ? "image/png"
                : MimetypesFileTypeMap.getDefaultFileTypeMap().getContentType(fileName);
        return uploadParams("public-read", prefix, fileName, contentType);
    }

    public FileUploadParams privatePutParams(String fileName, String prefix) {
        String contentType = MimetypesFileTypeMap.getDefaultFileTypeMap().getContentType(fileName);
        return uploadParams("private", prefix, fileName, contentType);
    }

    private FileUploadParams uploadParams(String acl, String prefix, String fileName, String contentType) {
        // Header Map
        Date date = new Date();
        Map<String, String> headerMap = new HashMap<>();
        headerMap.put("Host", this.getHost());
        headerMap.put("Date", getDate(date));
        headerMap.put("x-amz-content-sha256", "UNSIGNED-PAYLOAD");
        headerMap.put("x-amz-date", getAmzDate(date));
        headerMap.put("x-amz-storage-class", StorageClass);
        headerMap.put("x-amz-acl", acl);
        headerMap.put("Content-Type", contentType);
        // build params
        FileUploadParams params = new FileUploadParams();
        params.setHeaders(headerMap);
        params.setUri(this.buildSourceUri(prefix, fileName));
        params.setUrl("https://" + getHost() + "/" + bucket + params.getUri());
        params.setMethod("PUT");
        // System.out.println(params);
        String calculateSignature = calculateSignature(params);
        String currentDate = params.getHeaders().get("x-amz-date").substring(0, 8);
        String authorization = String.format(AuthorizationTemp, accessKey, currentDate, region, calculateSignature);
        params.getHeaders().put("Authorization", authorization);
        System.out.println(params);
        return params;
    }

    private String canonicalRequest(FileUploadParams uploadParams) {
        String canonicalRequest = uploadParams.getMethod()
                + "\n/" + bucket + uploadParams.getUri()
                + "\n"
                + "\ndate:" + uploadParams.getHeaders().get("Date")
                + "\nhost:" + uploadParams.getHeaders().get("Host")
                + "\nx-amz-content-sha256:" + uploadParams.getHeaders().get("x-amz-content-sha256")
                + "\nx-amz-date:" + uploadParams.getHeaders().get("x-amz-date")
                + "\nx-amz-storage-class:" + uploadParams.getHeaders().get("x-amz-storage-class")
                + "\n"
                + "\ndate;host;x-amz-content-sha256;x-amz-date;x-amz-storage-class"
                + "\n" + uploadParams.getHeaders().get("x-amz-content-sha256");
        System.out.println("\n canonicalRequest >>>  " + canonicalRequest);
        return canonicalRequest;
    }

    private String signString(FileUploadParams uploadParams) {
        String xAmzDate = uploadParams.getHeaders().get("x-amz-date");
        String currentDate = xAmzDate.substring(0, 8);
        String signString = HMACAlgorithm
                + "\n" + xAmzDate
                + "\n" + currentDate + "/" + region + "/" + ServiceName + "/" + Aws4Request
                + "\n" + generateHex(canonicalRequest(uploadParams));
        System.out.println("\n signString >>>  " + signString);
        return signString;
    }

    private String calculateSignature(FileUploadParams uploadParams) {
        String currentDate = uploadParams.getHeaders().get("x-amz-date").substring(0, 8);
        try {
            byte[] signatureKey = getSignatureKey(accessSecret, currentDate);
            byte[] signature = HmacSHA256(signatureKey, signString(uploadParams));
            return bytesToHex(signature);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return null;
    }

    private byte[] HmacSHA256(byte[] key, String data) throws Exception {
        String algorithm = "HmacSHA256";
        Mac mac = Mac.getInstance(algorithm);
        mac.init(new SecretKeySpec(key, algorithm));
        return mac.doFinal(data.getBytes(StandardCharsets.UTF_8));
    }

    private byte[] getSignatureKey(String key, String date) throws Exception {
        byte[] kSecret = ("AWS4" + key).getBytes(StandardCharsets.UTF_8);
        byte[] kDate = HmacSHA256(kSecret, date);
        byte[] kRegion = HmacSHA256(kDate, region);
        byte[] kService = HmacSHA256(kRegion, ServiceName);
        return HmacSHA256(kService, Aws4Request);
    }

    private String generateHex(String data) {
        return generateHex(
                data.getBytes(StandardCharsets.UTF_8)
        );
    }

    private String generateHex(byte[] data) {
        try {
            MessageDigest messageDigest = MessageDigest.getInstance("SHA-256");
            messageDigest.update(data);
            byte[] digest = messageDigest.digest();
            return String.format("%064x", new java.math.BigInteger(1, digest));
        }
        catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        return null;
    }

    final protected static char[] hexArray = "0123456789ABCDEF".toCharArray();

    private String bytesToHex(byte[] bytes) {
        char[] hexChars = new char[bytes.length * 2];
        for (int j = 0; j < bytes.length; j++) {
            int v = bytes[j] & 0xFF;
            hexChars[j * 2] = hexArray[v >>> 4];
            hexChars[j * 2 + 1] = hexArray[v & 0x0F];
        }
        return new String(hexChars).toLowerCase();
    }

    private String getAmzDate(Date date) {
        DateFormat dateFormat = new SimpleDateFormat("yyyyMMdd'T'HHmmss'Z'");
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        return dateFormat.format(date);
    }

    protected String getDate(Date date) {
        DateFormat dateFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss", Locale.ENGLISH);
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        return dateFormat.format(date) + " GMT";
    }

    private String encodeParameter(String param){
        try {
            return URLEncoder.encode(param, "UTF-8");
        } catch (Exception e) {
            return URLEncoder.encode(param);
        }
    }

    protected String getHost() {
        String endPoint = this.endPoint.startsWith("http")
                ? this.endPoint.substring(this.endPoint.indexOf("://") + 3)
                : this.endPoint;
        return endPoint;
    }

    protected String buildSourceUri(String prefix, String fileName) {
        prefix = prefix.startsWith("/") ? prefix.substring(1) : prefix;
        // 如果含有文件类型，prefix 就是完整文件路径
        if (prefix.contains(".")) {
            return "/" + prefix;
        }
        String suffix = fileName.substring(fileName.lastIndexOf(".")).toLowerCase();
        String[] dateSplit = new SimpleDateFormat("yyyy-MM-dd").format(new Date()).split("-");
        return "/" + prefix +
                "/" + dateSplit[0] +
                "/" + dateSplit[1] +
                "/" + UUID.randomUUID() + suffix;
    }

    private String bytesSHA256(byte[] bytes){
        MessageDigest messageDigest;
        String sha256Str = null;
        try {
            messageDigest = MessageDigest.getInstance("SHA-256");
            messageDigest.update(bytes);
            sha256Str = bytesToHex(messageDigest.digest());
        }
        catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        return sha256Str;
    }

    public byte[] readBytes(InputStream inputStream) {
        try {
            ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
            int _byte;
            while ((_byte = inputStream.read()) != -1)
                byteArrayOutputStream.write(_byte);
            byte[] result = byteArrayOutputStream.toByteArray();
            byteArrayOutputStream.close();
            inputStream.close();
            return result;
        } catch (Exception ex) {
            ex.printStackTrace();
            return null;
        }
    }
}


