package com.eworkcloud.web.util;

import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.alidns.model.v20150109.*;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import com.eworkcloud.dns.DnsClientConfiguration;

public abstract class DnsClientUtils {

    /**
     * 获取Acs客户端
     *
     * @return Acs客户端
     */
    public static IAcsClient getAcsClient() {
        IClientProfile profile = DefaultProfile.getProfile(DnsClientConfiguration.getRegionId(),
                DnsClientConfiguration.getAccessKeyId(), DnsClientConfiguration.getAccessKeySecret());
        return new DefaultAcsClient(profile);
    }

    /**
     * 获取解析记录
     *
     * @param recordId 记录ID
     * @return 解析记录
     */
    public static DescribeDomainRecordInfoResponse getDomain(String recordId) {
        IAcsClient acsClient = null;
        try {
            acsClient = getAcsClient();

            DescribeDomainRecordInfoRequest request = new DescribeDomainRecordInfoRequest();
            request.setRecordId(recordId);

            return acsClient.getAcsResponse(request);
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        } finally {
            if (null != acsClient) {
                acsClient.shutdown();
            }
        }
    }

    /**
     * 新增解析记录
     *
     * @param request 解析记录
     * @return RecordId
     */
    public static String addDomain(AddDomainRecordRequest request) {
        IAcsClient acsClient = null;
        try {
            acsClient = getAcsClient();

            // 检测输入的参数
            if (null == request.getType()) {
                request.setType("A");
            }
            if (null == request.getTTL()) {
                request.setTTL(600L);
            }
            if (null == request.getLine()) {
                request.setLine("default");
            }

            return acsClient.getAcsResponse(request).getRecordId();
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        } finally {
            if (null != acsClient) {
                acsClient.shutdown();
            }
        }
    }

    /**
     * 修改解析记录
     *
     * @param request 解析记录
     */
    public static void updateDomain(UpdateDomainRecordRequest request) {
        IAcsClient acsClient = null;
        try {
            acsClient = getAcsClient();
            acsClient.getAcsResponse(request);
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        } finally {
            if (null != acsClient) {
                acsClient.shutdown();
            }
        }
    }

    /**
     * 删除解析记录
     *
     * @param recordId 记录ID
     */
    public static void deleteDomain(String recordId) {
        IAcsClient acsClient = null;
        try {
            acsClient = getAcsClient();

            DeleteDomainRecordRequest request = new DeleteDomainRecordRequest();
            request.setRecordId(recordId);

            acsClient.getAcsResponse(request);
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        } finally {
            if (null != acsClient) {
                acsClient.shutdown();
            }
        }
    }

    /**
     * 设置解析状态（Enable: 启用解析 Disable: 暂停解析）
     *
     * @param recordId 记录ID
     * @param status   解析状态
     */
    public static void setStatus(String recordId, String status) {
        IAcsClient acsClient = null;
        try {
            acsClient = getAcsClient();

            SetDomainRecordStatusRequest request = new SetDomainRecordStatusRequest();
            request.setRecordId(recordId);
            request.setStatus(status);

            acsClient.getAcsResponse(request);
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        } finally {
            if (null != acsClient) {
                acsClient.shutdown();
            }
        }
    }

    /**
     * 查询解析列表
     *
     * @param request 解析记录
     * @return 解析列表
     */
    public static DescribeDomainsResponse queryDomains(DescribeDomainsRequest request) {
        IAcsClient acsClient = null;
        try {
            acsClient = getAcsClient();
            return acsClient.getAcsResponse(request);
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        } finally {
            if (null != acsClient) {
                acsClient.shutdown();
            }
        }
    }
}
