package com.eworkcloud.mybatis;

import com.github.pagehelper.PageInterceptor;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.session.SqlSessionFactory;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

@Slf4j
@Configuration
@EnableConfigurationProperties(MybatisProperties.class)
public class MybatisConfiguration {

    private static ResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();

    @javax.annotation.Resource
    private MybatisProperties mybatisProperties;

    private Resource[] resolveMapperLocations() {
        if (StringUtils.isEmpty(mybatisProperties.getMapperLocations())) {
            return new Resource[0];
        }
        String[] locations = mybatisProperties.getMapperLocations().split(",");
        List<Resource> resources = new ArrayList<>();
        for (String location : locations) {
            if (!StringUtils.isEmpty(location)) {
                try {
                    Collections.addAll(resources, resolver.getResources(location.trim()));
                } catch (IOException ex) {
                    log.warn(ex.getMessage());
                }
            }
        }
        return resources.toArray(new Resource[0]);
    }

    @Bean(destroyMethod = "close")
    public HikariDataSource hikariDataSource() {
        HikariConfig config = new HikariConfig();
        config.setDriverClassName(mybatisProperties.getDriverClassName());
        config.setJdbcUrl(mybatisProperties.getJdbcUrl());
        config.setUsername(mybatisProperties.getUsername());
        config.setPassword(mybatisProperties.getPassword());
        config.setMaximumPoolSize(mybatisProperties.getMaxPoolSize());
        config.setMaxLifetime(mybatisProperties.getMaxLifetime());

        config.addDataSourceProperty("cachePrepStmts", "true");
        config.addDataSourceProperty("prepStmtCacheSize", "250");
        config.addDataSourceProperty("prepStmtCacheSqlLimit", "2048");
        config.addDataSourceProperty("useServerPrepStmts", "true");
        config.addDataSourceProperty("useLocalSessionState", "true");
        config.addDataSourceProperty("useLocalTransactionState", "true");
        config.addDataSourceProperty("rewriteBatchedStatements", "true");
        config.addDataSourceProperty("cacheResultSetMetadata", "true");
        config.addDataSourceProperty("cacheServerConfiguration", "true");
        config.addDataSourceProperty("elideSetAutoCommits", "true");
        config.addDataSourceProperty("maintainTimeStats", "false");
        return new HikariDataSource(config);
    }

    @Bean
    public PageInterceptor pageInterceptor() {
        PageInterceptor pageInterceptor = new PageInterceptor();
        Properties properties = new Properties();
        properties.put("offsetAsPageNum", true);
        properties.put("rowBoundsWithCount", true);
        properties.put("reasonable", false);
        pageInterceptor.setProperties(properties);
        return pageInterceptor;
    }

    @Bean
    public SqlSessionFactory sqlSessionFactory(HikariDataSource hikariDataSource, PageInterceptor pageInterceptor) throws Exception {
        SqlSessionFactoryBean factoryBean = new SqlSessionFactoryBean();
        factoryBean.setDataSource(hikariDataSource);
        try {
            factoryBean.setMapperLocations(resolveMapperLocations());
            factoryBean.setTypeAliasesPackage(mybatisProperties.getTypeAliasesPackage());
        } catch (Exception ex) {
            log.warn(ex.getMessage());
        }
        factoryBean.setPlugins(new Interceptor[]{pageInterceptor});
        return factoryBean.getObject();
    }
}
