package com.eworkcloud.oss;

import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.CannedAccessControlList;
import com.aliyun.oss.model.CreateBucketRequest;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.ObjectListing;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.auth.sts.AssumeRoleRequest;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import org.apache.commons.io.IOUtils;
import org.springframework.data.redis.core.RedisTemplate;

import java.io.*;
import java.net.URL;
import java.util.Date;
import java.util.concurrent.TimeUnit;

import static com.aliyuncs.auth.sts.AssumeRoleResponse.Credentials;
import static com.eworkcloud.oss.OssClientProperties.Sts;

/**
 * OSS客户端
 *
 * @author 马恩全
 */
public class OssClientTemplate {
    private static final String REDIS_ALIYUN_TOKEN = "oauth:aliyun:";

    private OssClientProperties ossClientProperties;
    private RedisTemplate<String, Credentials> redisTemplate;

    public OssClientTemplate(OssClientProperties ossClientProperties, RedisTemplate<String, Credentials> redisTemplate) {
        this.ossClientProperties = ossClientProperties;
        this.redisTemplate = redisTemplate;
    }

    private Credentials getCredentials(Sts sts) throws Exception {
        Credentials credentials = redisTemplate.opsForValue().get(REDIS_ALIYUN_TOKEN + sts.getRoleSessionName());
        if (null != credentials) {
            return credentials;
        }

        DefaultProfile.addEndpoint("", "Sts", sts.getEndpoint());
        IClientProfile profile = DefaultProfile.getProfile("", sts.getAccessKeyId(), sts.getAccessKeySecret());
        IAcsClient acsClient = new DefaultAcsClient(profile);

        AssumeRoleRequest request = new AssumeRoleRequest();
        request.setRoleArn(sts.getRoleArn());
        request.setRoleSessionName(sts.getRoleSessionName());
        request.setDurationSeconds(sts.getDurationSeconds());
        credentials = acsClient.getAcsResponse(request).getCredentials();

        redisTemplate.opsForValue().set(REDIS_ALIYUN_TOKEN + sts.getRoleSessionName(), credentials,
                sts.getDurationSeconds() - 600L, TimeUnit.SECONDS);

        return credentials;
    }

    /**
     * 获取OSS客户端
     *
     * @return OSS客户端
     * @throws Exception 获取客户端异常
     */
    @SuppressWarnings("deprecation")
    public OSSClient getOSSClient() throws Exception {
        OSSClient ossClient;

        Sts sts = ossClientProperties.getSts();
        if (null != sts) {
            Credentials credentials = getCredentials(sts);
            ossClient = new OSSClient(ossClientProperties.getEndpoint(), credentials.getAccessKeyId(),
                    credentials.getAccessKeySecret(), credentials.getSecurityToken());
        } else {
            ossClient = new OSSClient(ossClientProperties.getEndpoint(), ossClientProperties.getAccessKeyId(),
                    ossClientProperties.getAccessKeySecret());
        }
        return ossClient;
    }

    /**
     * 检测并创建存储空间
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     */
    public void createBucket(OSSClient ossClient, String bucketName) {
        if (!ossClient.doesBucketExist(bucketName)) {
            ossClient.createBucket(bucketName);
            CreateBucketRequest createBucketRequest = new CreateBucketRequest(bucketName);
            createBucketRequest.setCannedACL(CannedAccessControlList.PublicRead);
            ossClient.createBucket(createBucketRequest);
        }
    }

    /**
     * 检测并创建存储空间
     *
     * @param ossClient OSS客户端
     */
    public void createBucket(OSSClient ossClient) {
        createBucket(ossClient, ossClientProperties.getBucketName());
    }

    /**
     * 下载指定键的对象
     *
     * @param bucketName 存储空间
     * @param key        键
     * @return 文件字节数组
     */
    public byte[] getObject(String bucketName, String key) {
        OSSClient ossClient = null;
        try {
            ossClient = getOSSClient();
            OSSObject ossObject = ossClient.getObject(bucketName, key);
            return IOUtils.toByteArray(ossObject.getObjectContent());
        } catch (Exception ex) {
            throw new RuntimeException(ex.getMessage());
        } finally {
            if (null != ossClient) {
                ossClient.shutdown();
            }
        }
    }

    /**
     * 下载指定键的对象
     *
     * @param key 键
     * @return 文件字节数组
     */
    public byte[] getObject(String key) {
        return getObject(ossClientProperties.getBucketName(), key);
    }

    /**
     * 上传指定键的对象
     *
     * @param bucketName 存储空间
     * @param key        键
     * @param url        URL
     */
    public void putObject(String bucketName, String key, URL url) {
        try {
            putObject(bucketName, key, url.openStream());
        } catch (IOException ex) {
            throw new RuntimeException(ex.getMessage());
        }
    }

    /**
     * 上传指定键的对象
     *
     * @param key 键
     * @param url URL
     */
    public void putObject(String key, URL url) {
        try {
            putObject(key, url.openStream());
        } catch (IOException ex) {
            throw new RuntimeException(ex.getMessage());
        }
    }

    /**
     * 上传指定键的对象
     *
     * @param bucketName 存储空间
     * @param key        键
     * @param file       文件
     */
    public void putObject(String bucketName, String key, File file) {
        try {
            putObject(bucketName, key, new FileInputStream(file));
        } catch (FileNotFoundException ex) {
            throw new RuntimeException(ex.getMessage());
        }
    }

    /**
     * 上传指定键的对象
     *
     * @param key  键
     * @param file 文件
     */
    public void putObject(String key, File file) {
        try {
            putObject(key, new FileInputStream(file));
        } catch (FileNotFoundException ex) {
            throw new RuntimeException(ex.getMessage());
        }
    }

    /**
     * 上传指定键的对象
     *
     * @param bucketName 存储空间
     * @param key        键
     * @param input      文件流
     */
    public void putObject(String bucketName, String key, byte[] input) {
        putObject(bucketName, key, new ByteArrayInputStream(input));
    }

    /**
     * 上传指定键的对象
     *
     * @param key   键
     * @param input 文件流
     */
    public void putObject(String key, byte[] input) {
        putObject(key, new ByteArrayInputStream(input));
    }

    /**
     * 上传指定键的对象
     *
     * @param bucketName 存储空间
     * @param key        键
     * @param input      文件流
     */
    public void putObject(String bucketName, String key, InputStream input) {
        OSSClient ossClient = null;
        try {
            ossClient = getOSSClient();
            ossClient.putObject(bucketName, key, input);
        } catch (Exception ex) {
            throw new RuntimeException(ex.getMessage());
        } finally {
            if (null != ossClient) {
                ossClient.shutdown();
            }
        }
    }

    /**
     * 上传指定键的对象
     *
     * @param key   键
     * @param input 文件流
     */
    public void putObject(String key, InputStream input) {
        putObject(ossClientProperties.getBucketName(), key, input);
    }

    /**
     * 删除指定键的对象
     *
     * @param bucketName 存储空间
     * @param key        键
     */
    public void deleteObject(String bucketName, String key) {
        OSSClient ossClient = null;
        try {
            ossClient = getOSSClient();
            ossClient.deleteObject(bucketName, key);
        } catch (Exception ex) {
            throw new RuntimeException(ex.getMessage());
        } finally {
            if (null != ossClient) {
                ossClient.shutdown();
            }
        }
    }

    /**
     * 删除指定键的对象
     *
     * @param key 键
     */
    public void deleteObject(String key) {
        deleteObject(ossClientProperties.getBucketName(), key);
    }

    /**
     * 检测文件是否存在
     *
     * @param bucketName 存储空间
     * @param key        键
     * @return 是否存在
     */
    public boolean doesObjectExist(String bucketName, String key) {
        OSSClient ossClient = null;
        try {
            ossClient = getOSSClient();
            return ossClient.doesObjectExist(bucketName, key);
        } catch (Exception ex) {
            throw new RuntimeException(ex.getMessage());
        } finally {
            if (null != ossClient) {
                ossClient.shutdown();
            }
        }
    }

    /**
     * 检测文件是否存在
     *
     * @param key 键
     * @return 是否存在
     */
    public boolean doesObjectExist(String key) {
        return doesObjectExist(ossClientProperties.getBucketName(), key);
    }

    /**
     * 列举存储空间下指定前缀的文件。最多列举100个文件。
     *
     * @param bucketName 存储空间
     * @param prefix     前缀
     * @return 文件列表
     */
    public ObjectListing listObjects(String bucketName, String prefix) {
        OSSClient ossClient = null;
        try {
            ossClient = getOSSClient();
            return ossClient.listObjects(bucketName, prefix);
        } catch (Exception ex) {
            throw new RuntimeException(ex.getMessage());
        } finally {
            if (null != ossClient) {
                ossClient.shutdown();
            }
        }
    }

    /**
     * 列举存储空间下指定前缀的文件。最多列举100个文件。
     *
     * @param prefix 前缀
     * @return 文件列表
     */
    public ObjectListing listObjects(String prefix) {
        return listObjects(ossClientProperties.getBucketName(), prefix);
    }

    /**
     * 返回GET签名访问的URL
     *
     * @param bucketName 存储空间
     * @param key        键
     * @param expiration 到期时间
     * @return 签名访问的URL
     */
    public URL generatePresignedUrl(String bucketName, String key, Date expiration) {
        OSSClient ossClient = null;
        try {
            ossClient = getOSSClient();
            return ossClient.generatePresignedUrl(bucketName, key, expiration);
        } catch (Exception ex) {
            throw new RuntimeException(ex.getMessage());
        } finally {
            if (null != ossClient) {
                ossClient.shutdown();
            }
        }
    }

    /**
     * 返回GET签名访问的URL
     * expiration = 1000 * 3600 * 24
     *
     * @param bucketName 存储空间
     * @param key        键
     * @return 签名访问的URL
     */
    public URL generatePresignedUrl(String bucketName, String key) {
        OSSClient ossClient = null;
        try {
            ossClient = getOSSClient();
            Date expiration = new Date(new Date().getTime() + 1000 * 3600 * 24);
            return ossClient.generatePresignedUrl(bucketName, key, expiration);
        } catch (Exception ex) {
            throw new RuntimeException(ex.getMessage());
        } finally {
            if (null != ossClient) {
                ossClient.shutdown();
            }
        }
    }

    /**
     * 返回GET签名访问的URL
     *
     * @param key        键
     * @param expiration 到期时间
     * @return 签名访问的URL
     */
    public URL generatePresignedUrl(String key, Date expiration) {
        return generatePresignedUrl(ossClientProperties.getBucketName(), key, expiration);
    }

    /**
     * 返回GET签名访问的URL
     * expiration = 1000 * 3600 * 24
     *
     * @param key 键
     * @return 签名访问的URL
     */
    public URL generatePresignedUrl(String key) {
        return generatePresignedUrl(ossClientProperties.getBucketName(), key);
    }
}
