package com.eworkcloud.web.util;

import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.CannedAccessControlList;
import com.aliyun.oss.model.CreateBucketRequest;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.ObjectListing;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.auth.sts.AssumeRoleRequest;
import com.aliyuncs.auth.sts.AssumeRoleResponse.Credentials;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import com.eworkcloud.oss.OssClientConfiguration;
import org.apache.commons.io.IOUtils;

import java.io.*;
import java.net.URL;
import java.util.Date;

public abstract class OssClientUtils {

    /**
     * 获取OSS证书（STS）
     *
     * @return OSS证书
     */
    public static Credentials getCredentials() {
        try {
            DefaultProfile.addEndpoint("", "Sts", OssClientConfiguration.getEndpoint());
            IClientProfile profile = DefaultProfile.getProfile("",
                    OssClientConfiguration.getAccessKeyId(), OssClientConfiguration.getAccessKeySecret());
            IAcsClient acsClient = new DefaultAcsClient(profile);

            AssumeRoleRequest request = new AssumeRoleRequest();
            request.setRoleArn(OssClientConfiguration.getRoleArn());
            request.setRoleSessionName(OssClientConfiguration.getRoleSessionName());
            request.setDurationSeconds(OssClientConfiguration.getDurationSeconds());

            return acsClient.getAcsResponse(request).getCredentials();
        } catch (ClientException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * 获取OSS客户端（STS）
     *
     * @param credentials OSS证书
     * @return OSS客户端
     */
    @SuppressWarnings("deprecation")
    public static OSSClient getOSSClient(Credentials credentials) {
        return new OSSClient(OssClientConfiguration.getEndpoint(), credentials.getAccessKeyId(),
                credentials.getAccessKeySecret(), credentials.getSecurityToken());
    }

    /**
     * 获取OSS客户端（OSS）
     *
     * @return OSS客户端
     */
    @SuppressWarnings("deprecation")
    public static OSSClient getOSSClient() {
        return new OSSClient(OssClientConfiguration.getEndpoint(), OssClientConfiguration.getAccessKeyId(),
                OssClientConfiguration.getAccessKeySecret());
    }

    /**
     * 检测并创建存储空间
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     */
    public static void createBucket(OSSClient ossClient, String bucketName) {
        if (!ossClient.doesBucketExist(bucketName)) {
            ossClient.createBucket(bucketName);
            CreateBucketRequest createBucketRequest = new CreateBucketRequest(bucketName);
            createBucketRequest.setCannedACL(CannedAccessControlList.PublicRead);
            ossClient.createBucket(createBucketRequest);
        }
    }

    /**
     * 检测并创建存储空间
     *
     * @param ossClient OSS客户端
     */
    public static void createBucket(OSSClient ossClient) {
        createBucket(ossClient, OssClientConfiguration.getBucketName());
    }

    /**
     * 下载指定键的对象
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     * @param key        键
     * @return 文件字节数组
     */
    public static byte[] getObject(OSSClient ossClient, String bucketName, String key) {
        try {
            OSSObject ossObject = ossClient.getObject(bucketName, key);
            return IOUtils.toByteArray(ossObject.getObjectContent());
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * 下载指定键的对象
     *
     * @param ossClient OSS客户端
     * @param key       键
     * @return 文件字节数组
     */
    public static byte[] getObject(OSSClient ossClient, String key) {
        return getObject(ossClient, OssClientConfiguration.getBucketName(), key);
    }

    /**
     * 上传指定键的对象
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     * @param key        键
     * @param url        URL
     */
    public static void putObject(OSSClient ossClient, String bucketName, String key, URL url) {
        try {
            putObject(ossClient, bucketName, key, url.openStream());
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * 上传指定键的对象
     *
     * @param ossClient OSS客户端
     * @param key       键
     * @param url       URL
     */
    public static void putObject(OSSClient ossClient, String key, URL url) {
        try {
            putObject(ossClient, key, url.openStream());
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * 上传指定键的对象
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     * @param key        键
     * @param file       文件
     */
    public static void putObject(OSSClient ossClient, String bucketName, String key, File file) {
        try {
            putObject(ossClient, bucketName, key, new FileInputStream(file));
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * 上传指定键的对象
     *
     * @param ossClient OSS客户端
     * @param key       键
     * @param file      文件
     */
    public static void putObject(OSSClient ossClient, String key, File file) {
        try {
            putObject(ossClient, key, new FileInputStream(file));
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * 上传指定键的对象
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     * @param key        键
     * @param input      文件流
     */
    public static void putObject(OSSClient ossClient, String bucketName, String key, byte[] input) {
        putObject(ossClient, bucketName, key, new ByteArrayInputStream(input));
    }

    /**
     * 上传指定键的对象
     *
     * @param ossClient OSS客户端
     * @param key       键
     * @param input     文件流
     */
    public static void putObject(OSSClient ossClient, String key, byte[] input) {
        putObject(ossClient, key, new ByteArrayInputStream(input));
    }

    /**
     * 上传指定键的对象
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     * @param key        键
     * @param input      文件流
     */
    public static void putObject(OSSClient ossClient, String bucketName, String key, InputStream input) {
        ossClient.putObject(bucketName, key, input);
    }

    /**
     * 上传指定键的对象
     *
     * @param ossClient OSS客户端
     * @param key       键
     * @param input     文件流
     */
    public static void putObject(OSSClient ossClient, String key, InputStream input) {
        putObject(ossClient, OssClientConfiguration.getBucketName(), key, input);
    }

    /**
     * 删除指定键的对象
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     * @param key        键
     */
    public static void deleteObject(OSSClient ossClient, String bucketName, String key) {
        ossClient.deleteObject(bucketName, key);
    }

    /**
     * 删除指定键的对象
     *
     * @param ossClient OSS客户端
     * @param key       键
     */
    public static void deleteObject(OSSClient ossClient, String key) {
        deleteObject(ossClient, OssClientConfiguration.getBucketName(), key);
    }

    /**
     * 检测文件是否存在
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     * @param key        键
     * @return 是否存在
     */
    public static boolean doesObjectExist(OSSClient ossClient, String bucketName, String key) {
        return ossClient.doesObjectExist(bucketName, key);
    }

    /**
     * 检测文件是否存在
     *
     * @param ossClient OSS客户端
     * @param key       键
     * @return 是否存在
     */
    public static boolean doesObjectExist(OSSClient ossClient, String key) {
        return doesObjectExist(ossClient, OssClientConfiguration.getBucketName(), key);
    }

    /**
     * 列举存储空间下指定前缀的文件。最多列举100个文件。
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     * @param prefix     前缀
     * @return 文件列表
     */
    public static ObjectListing listObjects(OSSClient ossClient, String bucketName, String prefix) {
        return ossClient.listObjects(bucketName, prefix);
    }

    /**
     * 列举存储空间下指定前缀的文件。最多列举100个文件。
     *
     * @param ossClient OSS客户端
     * @param prefix    前缀
     * @return 文件列表
     */
    public static ObjectListing listObjects(OSSClient ossClient, String prefix) {
        return listObjects(ossClient, OssClientConfiguration.getBucketName(), prefix);
    }

    /**
     * 返回GET签名访问的URL
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     * @param key        键
     * @param expiration 到期时间
     * @return 签名访问的URL
     */
    public static URL generatePresignedUrl(OSSClient ossClient, String bucketName, String key, Date expiration) {
        return ossClient.generatePresignedUrl(bucketName, key, expiration);
    }

    /**
     * 返回GET签名访问的URL
     * expiration = 1000 * 3600 * 24
     *
     * @param ossClient  OSS客户端
     * @param bucketName 存储空间
     * @param key        键
     * @return 签名访问的URL
     */
    public static URL generatePresignedUrl(OSSClient ossClient, String bucketName, String key) {
        Date expiration = new Date(new Date().getTime() + 1000 * 3600 * 24);
        return generatePresignedUrl(ossClient, bucketName, key, expiration);
    }

    /**
     * 返回GET签名访问的URL
     *
     * @param ossClient  OSS客户端
     * @param key        键
     * @param expiration 到期时间
     * @return 签名访问的URL
     */
    public static URL generatePresignedUrl(OSSClient ossClient, String key, Date expiration) {
        return generatePresignedUrl(ossClient, OssClientConfiguration.getBucketName(), key, expiration);
    }

    /**
     * 返回GET签名访问的URL
     * expiration = 1000 * 3600 * 24
     *
     * @param ossClient OSS客户端
     * @param key       键
     * @return 签名访问的URL
     */
    public static URL generatePresignedUrl(OSSClient ossClient, String key) {
        return generatePresignedUrl(ossClient, OssClientConfiguration.getBucketName(), key);
    }
}
