package com.eworkcloud.redis;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.util.Arrays;
import java.util.Set;
import java.util.concurrent.TimeUnit;

@Slf4j
public class RedisCacheTemplate {

    private StringRedisTemplate stringRedisTemplate;
    private ObjectMapper objectMapper;

    protected RedisCacheTemplate(StringRedisTemplate stringRedisTemplate, ObjectMapper objectMapper) {
        this.stringRedisTemplate = stringRedisTemplate;
        this.objectMapper = objectMapper;
    }

    public String serialize(Object value) {
        try {
            if (null != value) {
                return objectMapper.writeValueAsString(value);
            } else {
                return null;
            }
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    @SuppressWarnings("unchecked")
    public <T> T deserialize(String value) {
        try {
            if (StringUtils.hasText(value)) {
                return (T) objectMapper.readValue(value, Object.class);
            } else {
                return null;
            }
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    /************************************* opsForRedis ********************************************/

    /**
     * 设置过期时间（-1 不过期）
     *
     * @param key  键¬
     * @param time 时间(秒)
     * @return 是否成功（false 不存在）
     */
    public boolean expire(String key, long time) {
        return stringRedisTemplate.expire(key, time, TimeUnit.SECONDS);
    }

    /**
     * 获取过期时间（-1 不过期、-2 不存在）
     *
     * @param key 键
     * @return 时间(秒)
     */
    public Long getExpire(String key) {
        return stringRedisTemplate.getExpire(key, TimeUnit.SECONDS);
    }

    /**
     * 判断key是否存在
     *
     * @param key 键
     * @return 是否存在
     */
    public boolean hasKey(String key) {
        return stringRedisTemplate.hasKey(key);
    }

    /**
     * 获取REDIS键集
     *
     * @param pattern 条件
     * @return 键集合
     */
    public Set<String> keys(String pattern) {
        return stringRedisTemplate.keys(pattern);
    }

    /**
     * 删除REDIS缓存
     *
     * @param keys 键
     */
    public void delete(String... keys) {
        if (!ObjectUtils.isEmpty(keys)) {
            if (keys.length == 1) {
                stringRedisTemplate.delete(keys[0]);
            } else {
                stringRedisTemplate.delete(Arrays.asList(keys));
            }
        }
    }

    /************************************* opsForValue ********************************************/

    /**
     * 获取REDIS值
     *
     * @param key 键
     * @return 返回值
     */
    public String getString(String key) {
        return stringRedisTemplate.opsForValue().get(key);
    }

    /**
     * 获取REDIS值
     *
     * @param key 键
     * @param <T> 返回类型
     * @return 返回值
     */
    public <T> T getObject(String key) {
        return deserialize(getString(key));
    }

    /**
     * 设置REDIS值
     *
     * @param key   键
     * @param value 值
     */
    public void setString(String key, String value) {
        setString(key, value, 0);
    }

    /**
     * 设置REDIS值
     *
     * @param key   键
     * @param value 值
     * @param time  时间(秒)
     */
    public void setString(String key, String value, long time) {
        if (time > 0) {
            stringRedisTemplate.opsForValue().set(key, value, time, TimeUnit.SECONDS);
        } else {
            stringRedisTemplate.opsForValue().set(key, value);
        }
    }

    /**
     * 设置REDIS值
     *
     * @param key   键
     * @param value 值
     */
    public void setObject(String key, Object value) {
        setObject(key, value, 0);
    }

    /**
     * 设置REDIS值
     *
     * @param key   键
     * @param value 值
     * @param time  时间(秒)
     */
    public void setObject(String key, Object value, long time) {
        setString(key, serialize(value), time);
    }

    /**
     * 递增
     *
     * @param key   键
     * @param delta 因子(大于0)
     * @return 返回值
     */
    public long increment(String key, long delta) {
        if (delta < 0) {
            throw new RuntimeException("递增因子必须大于0");
        }
        return stringRedisTemplate.opsForValue().increment(key, delta);
    }

    /**
     * 递减
     *
     * @param key   键
     * @param delta 因子(大于0)
     * @return 返回值
     */
    public long decrement(String key, long delta) {
        if (delta < 0) {
            throw new RuntimeException("递减因子必须大于0");
        }
        return stringRedisTemplate.opsForValue().increment(key, -delta);
    }
}
