package com.eworkcloud.web.util;

import com.eworkcloud.web.config.WxworkConfiguration;
import com.eworkcloud.web.enums.HttpMethod;
import com.eworkcloud.web.exception.BusinessException;
import com.eworkcloud.web.model.ByteArray;
import com.eworkcloud.web.model.HttpMessage;
import com.eworkcloud.web.wxwork.WxworkMessage;
import com.eworkcloud.web.wxwork.WxworkSendStatus;
import com.eworkcloud.web.wxwork.WxworkSession;
import com.eworkcloud.web.wxwork.WxworkTokenInfo;
import com.eworkcloud.web.wxwork.WxworkUserInfo;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.StringReader;
import java.util.Arrays;
import java.util.Map;

import static com.eworkcloud.web.util.Constants.CHARSET;

public abstract class WxworkUtils {
    /**
     * 获取访问令牌
     * ?corpid=ID&corpsecret=SECRET
     */
    private static final String ACCESS_TOKEN = "https://qyapi.weixin.qq.com/cgi-bin/gettoken";

    /**
     * CODE换登录者ID
     */
    private static final String CODE2USERINFO = "https://qyapi.weixin.qq.com/cgi-bin/user/getuserinfo";

    /***
     * CODE换会话KEY
     */
    private static final String CODE2SESSION = "https://qyapi.weixin.qq.com/cgi-bin/miniprogram/jscode2session";

    /**
     * 发送应用消息
     * ?access_token=ACCESS_TOKEN
     */
    private static final String SEND_MESSAGE = "https://qyapi.weixin.qq.com/cgi-bin/message/send";


    private static String encrypt(String text) {
        if (WxworkConfiguration.getAesKey().length() != 43) {
            throw new BusinessException("AesKey非法");
        }
        byte[] aesKey = Base64Utils.decodeToBytes(WxworkConfiguration.getAesKey() + "=");

        byte[] textBytes = text.getBytes(CHARSET);

        ByteArray byteArray = new ByteArray();
        // random + networkBytesOrder + text + corpid
        byteArray.addValue(WebUtils.randomString(16));
        byteArray.addValue(CryptoUtils.intToBytes(textBytes.length));
        byteArray.addValue(textBytes);
        byteArray.addValue(WxworkConfiguration.getCorpid());
        // PKCS7Padding
        byteArray.addValue(PKCS7Padding.encode(byteArray.size()));

        try {
            Cipher cipher = Cipher.getInstance("AES/CBC/NoPadding");
            SecretKeySpec keySpec = new SecretKeySpec(aesKey, "AES");
            IvParameterSpec iv = new IvParameterSpec(aesKey, 0, 16);
            cipher.init(Cipher.ENCRYPT_MODE, keySpec, iv);

            return Base64Utils.encodeToString(cipher.doFinal(byteArray.toBytes()));
        } catch (Exception ex) {
            throw new BusinessException("AES加密失败", ex);
        }
    }

    private static String decrypt(String text) {
        if (WxworkConfiguration.getAesKey().length() != 43) {
            throw new BusinessException("AesKey非法");
        }
        byte[] aesKey = Base64Utils.decodeToBytes(WxworkConfiguration.getAesKey() + "=");

        byte[] original;
        try {
            Cipher cipher = Cipher.getInstance("AES/CBC/NoPadding");
            SecretKeySpec keySpec = new SecretKeySpec(aesKey, "AES");
            IvParameterSpec iv = new IvParameterSpec(aesKey, 0, 16);
            cipher.init(Cipher.DECRYPT_MODE, keySpec, iv);

            original = cipher.doFinal(Base64Utils.decodeToBytes(text));
        } catch (Exception ex) {
            throw new BusinessException("AES解密失败", ex);
        }

        String xmlContent, corpid;
        try {
            byte[] bytes = PKCS7Padding.decode(original);

            int xmlLength = CryptoUtils.bytesToInt(Arrays.copyOfRange(bytes, 16, 20));
            xmlContent = new String(Arrays.copyOfRange(bytes, 20, 20 + xmlLength), CHARSET);

            corpid = new String(Arrays.copyOfRange(bytes, 20 + xmlLength, bytes.length), CHARSET);
        } catch (Exception ex) {
            throw new BusinessException("解密数据异常", ex);
        }

        if (!corpid.equals(WxworkConfiguration.getCorpid())) {
            throw new BusinessException("CorpID非法");
        }
        return xmlContent;
    }

    private static String extract(String text) {
        try {
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            dbf.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
            dbf.setFeature("http://xml.org/sax/features/external-general-entities", false);
            dbf.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
            dbf.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
            dbf.setXIncludeAware(false);
            dbf.setExpandEntityReferences(false);

            DocumentBuilder db = dbf.newDocumentBuilder();
            StringReader sr = new StringReader(text);
            InputSource is = new InputSource(sr);
            Document document = db.parse(is);

            Element root = document.getDocumentElement();
            NodeList encrypt = root.getElementsByTagName("Encrypt");

            return encrypt.item(0).getTextContent();
        } catch (Exception ex) {
            throw new BusinessException("XML解析失败", ex);
        }
    }

    private static String generate(String encrypt, String signature, String timestamp, String nonce) {
        String format = "<xml>\n"
                + "<Encrypt><![CDATA[%1$s]]></Encrypt>\n"
                + "<MsgSignature><![CDATA[%2$s]]></MsgSignature>\n"
                + "<TimeStamp>%3$s</TimeStamp>\n"
                + "<Nonce><![CDATA[%4$s]]></Nonce>\n"
                + "</xml>";
        return String.format(format, encrypt, signature, timestamp, nonce);
    }

    private static String signature(String timestamp, String nonce, String echostr) {
        try {
            String[] array = new String[]{WxworkConfiguration.getToken(), timestamp, nonce, echostr};
            Arrays.sort(array);

            StringBuilder builder = new StringBuilder();
            for (int i = 0; i < 4; i++) {
                builder.append(array[i]);
            }

            return CryptoUtils.shaDigest(builder.toString());
        } catch (Exception ex) {
            throw new BusinessException("SHA签名失败", ex);
        }
    }


    /**
     * 获取访问令牌
     *
     * @param secret 应用密钥
     * @return 访问令牌
     */
    public static WxworkTokenInfo accessToken(String secret) {
        return OkHttpUtils.execute(HttpMessage.builder()
                .url(ACCESS_TOKEN)
                .build()
                .addQuery("corpid", WxworkConfiguration.getCorpid())
                .addQuery("corpsecret", secret), WxworkTokenInfo.class);
    }

    /**
     * 获取访问令牌
     *
     * @return 访问令牌
     */
    public static WxworkTokenInfo accessToken() {
        return accessToken(WxworkConfiguration.getSecret());
    }


    /**
     * CODE获取登录者ID
     *
     * @param accessToken 访问令牌
     * @param code        认证CODE
     * @return 登录者ID
     */
    public static WxworkUserInfo code2UserInfo(String accessToken, String code) {
        return OkHttpUtils.execute(HttpMessage.builder()
                .url(CODE2USERINFO)
                .build()
                .addQuery("access_token", accessToken)
                .addQuery("code", code), WxworkUserInfo.class);
    }

    /**
     * CODE获取小程序会话
     *
     * @param accessToken 访问令牌
     * @param code        认证CODE
     * @return 登录者ID
     */
    public static WxworkSession code2Session(String accessToken, String code) {
        return OkHttpUtils.execute(HttpMessage.builder()
                .url(CODE2SESSION)
                .build()
                .addQuery("access_token", accessToken)
                .addQuery("js_code", code)
                .addQuery("grant_type", "authorization_code"), WxworkSession.class);
    }


    /**
     * 发送应用消息
     *
     * @param accessToken 访问令牌
     * @param formData    消息内容
     * @return 发送状态
     */
    public static WxworkSendStatus sendMessage(String accessToken, Map<String, Object> formData) {
        if (!formData.containsKey("agentid") || Assert.isEmpty(formData.get("agentid"))) {
            formData.put("agentid", WxworkConfiguration.getAgentid());
        }

        return OkHttpUtils.execute(HttpMessage.builder()
                .url(SEND_MESSAGE)
                .method(HttpMethod.POST)
                .formData(formData)
                .build()
                .addQuery("access_token", accessToken), WxworkSendStatus.class);
    }

    /**
     * 发送应用消息
     *
     * @param accessToken 访问令牌
     * @param message     消息内容
     * @return 发送状态
     */
    public static WxworkSendStatus sendMessage(String accessToken, WxworkMessage message) {
        return sendMessage(accessToken, OrikaUtils.beanToMap(message));
    }


    /**
     * 将回复用户的消息加密打包
     *
     * @param message   待回复的消息
     * @param timestamp 时间戳
     * @param nonce     随机串
     * @return 加密后的密文
     */
    public static String encrypt(String message, String timestamp, String nonce) {
        String encrypt = encrypt(message);

        String signature = signature(timestamp, nonce, encrypt);

        return generate(encrypt, signature, timestamp, nonce);
    }

    /**
     * 检验消息并获取解密后的明文
     *
     * @param signature 加密签名
     * @param timestamp 时间戳
     * @param nonce     随机串
     * @param postData  请求字符串
     * @return 解密后的明文
     */
    public static String decrypt(String signature, String timestamp, String nonce, String postData) {
        String echostr = extract(postData);

        String validate = signature(timestamp, nonce, echostr);
        if (!validate.equals(signature)) {
            throw new BusinessException("签名验证错误");
        }

        return decrypt(echostr);
    }

    /**
     * 验证URL有效性
     *
     * @param signature 加密签名
     * @param timestamp 时间戳
     * @param nonce     随机串
     * @param echostr   加密的字符串
     * @return 消息内容明文
     */
    public static String verify(String signature, String timestamp, String nonce, String echostr) {
        String validate = signature(timestamp, nonce, echostr);
        if (!validate.equals(signature)) {
            throw new BusinessException("签名验证错误");
        }

        return decrypt(echostr);
    }
}
