package com.eworkcloud.web.util;

import com.eworkcloud.web.config.BaiduTranslateConfig;
import com.eworkcloud.web.enums.ContentType;
import com.eworkcloud.web.enums.HttpMethod;
import com.eworkcloud.web.model.Translation;
import lombok.Getter;
import lombok.Setter;

import java.util.List;

public abstract class BaiduUtils {

    private static final String TRANSLATE = "https://api.fanyi.baidu.com/api/trans/vip/translate";

    @Getter
    @Setter
    private static class TransModel {
        /**
         * 源语言
         */
        private String from;
        /**
         * 目标语言
         */
        private String to;
        /**
         * 翻译列表
         */
        private List<TransText> trans_result;
    }

    @Getter
    @Setter
    private static class TransText {
        /**
         * 原文
         */
        private String src;
        /**
         * 译文
         */
        private String dst;
    }

    /**
     * 翻译单个文本
     *
     * @param text 文本
     * @param from 源语言
     * @param to   目标语言
     * @return 翻译结果
     */
    public static Translation translate(String text, String from, String to) {
        String appid = BaiduTranslateConfig.getAppid();
        String secret = BaiduTranslateConfig.getSecret();

        String salt = String.valueOf(System.currentTimeMillis());
        String sign = CryptoUtils.md5Digest(appid + text + salt + secret);

        OkHttpParam param = OkHttpParam.builder()
                .url(TRANSLATE)
                .method(HttpMethod.POST)
                .contentType(ContentType.FORM)
                .build();

        param.addFromData("q", WebUtils.urlEncoder(text));
        if (Assert.hasText(from)) {
            param.addFromData("from", from);
        }
        param.addFromData("to", to);
        param.addFromData("appid", appid);
        param.addFromData("salt", salt);
        param.addFromData("sign", sign);

        TransModel model = OkHttpUtils.execute(param, TransModel.class);
        if (Assert.notNull(model) && Assert.notEmpty(model.getTrans_result())) {
            Translation translation = new Translation();
            translation.setFrom(model.getFrom());
            translation.setTo(model.getTo());
            TransText transText = model.getTrans_result().get(0);
            translation.setSrc(transText.getSrc());
            translation.setDst(transText.getDst());
            return translation;
        } else {
            throw new RuntimeException("Failure to obtain baidu translation");
        }
    }

    /**
     * 翻译单个文本
     *
     * @param text 文本
     * @param to   目标语言
     * @return 翻译结果
     */
    public static Translation translate(String text, String to) {
        return translate(text, null, to);
    }
}
