/*
 * Decompiled with CFR 0.152.
 */
package com.exasol.projectkeeper.dependencyupdate;

import com.exasol.errorreporting.ExaError;
import com.exasol.projectkeeper.Logger;
import com.exasol.projectkeeper.dependencyupdate.XmlDocumentIO;
import com.exasol.projectkeeper.shared.config.ProjectKeeperConfig;
import com.exasol.projectkeeper.shared.config.ProjectKeeperModule;
import com.exasol.projectkeeper.shared.config.Source;
import com.exasol.projectkeeper.shared.config.VersionConfig;
import com.exasol.projectkeeper.shared.config.VersionFromSource;
import com.exasol.projectkeeper.sources.analyze.generic.CommandExecutor;
import com.exasol.projectkeeper.sources.analyze.generic.MavenProcessBuilder;
import com.exasol.projectkeeper.sources.analyze.generic.ShellCommand;
import com.exasol.projectkeeper.validators.changesfile.ChangesFile;
import com.exasol.projectkeeper.validators.changesfile.ChangesFileIO;
import com.vdurmont.semver4j.Semver;
import java.nio.file.Path;
import java.time.Clock;
import java.time.Duration;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

class ProjectVersionIncrementor {
    private static final ZoneId UTC_ZONE = ZoneId.of("UTC");
    private final ProjectKeeperConfig config;
    private final String currentProjectVersion;
    private final ChangesFileIO changesFileIO;
    private final Clock clock;
    private final Path projectDir;
    private final Logger logger;
    private final XmlDocumentIO xmlFileIO;
    private final CommandExecutor commandExecutor;

    ProjectVersionIncrementor(ProjectKeeperConfig config, Logger logger, Path projectDir, String currentProjectVersion) {
        this(config, logger, projectDir, currentProjectVersion, new ChangesFileIO(), new XmlDocumentIO(), new CommandExecutor(), Clock.systemUTC());
    }

    ProjectVersionIncrementor(ProjectKeeperConfig config, Logger logger, Path projectDir, String currentProjectVersion, ChangesFileIO changesFileIO, XmlDocumentIO xmlFileIO, CommandExecutor commandExecutor, Clock clock) {
        this.config = config;
        this.logger = logger;
        this.projectDir = projectDir;
        this.changesFileIO = changesFileIO;
        this.xmlFileIO = xmlFileIO;
        this.commandExecutor = commandExecutor;
        this.clock = clock;
        this.currentProjectVersion = Objects.requireNonNull(currentProjectVersion, "currentProjectVersion");
    }

    boolean isCurrentVersionReleased() {
        Path changesFilePath = this.projectDir.resolve(ChangesFile.getPathForVersion(this.currentProjectVersion));
        ChangesFile changesFile = this.changesFileIO.read(changesFilePath);
        Optional<LocalDate> releaseDate = changesFile.getParsedReleaseDate();
        if (releaseDate.isEmpty()) {
            this.logger.info("Found invalid date '" + changesFile.getReleaseDate() + "' in changelog " + changesFilePath + ": version " + this.currentProjectVersion + " was not yet released");
            return false;
        }
        boolean released = releaseDate.get().isBefore(this.today());
        if (released) {
            this.logger.info("Version " + this.currentProjectVersion + " was released on " + changesFile.getReleaseDate() + " according to " + changesFilePath);
        }
        return released;
    }

    private LocalDate today() {
        return LocalDate.ofInstant(this.clock.instant(), UTC_ZONE);
    }

    String incrementProjectVersion() {
        String nextVersion = ProjectVersionIncrementor.getIncrementedVersion(this.currentProjectVersion);
        this.updatePomVersion(nextVersion);
        this.sourcesUsingReferenceCheckerPlugin().forEach(this::updateReferences);
        return nextVersion;
    }

    private void updatePomVersion(String nextVersion) {
        Path path = this.getPomPath();
        this.logger.info("Incrementing version from " + this.currentProjectVersion + " to " + nextVersion + " in POM " + path);
        Document pom = this.xmlFileIO.read(path);
        this.incrementVersion(path, pom, nextVersion);
        this.xmlFileIO.write(pom, path);
    }

    private Stream<Source> sourcesUsingReferenceCheckerPlugin() {
        return this.config.getSources().stream().filter(source -> source.getModules().contains((Object)ProjectKeeperModule.JAR_ARTIFACT));
    }

    private void updateReferences(Source source) {
        Path moduleDir = this.projectDir.resolve(source.getPath()).getParent();
        this.logger.info("Unify artifact references in dir " + moduleDir + "...");
        ShellCommand command = MavenProcessBuilder.create().addArgument("artifact-reference-checker:unify").workingDir(moduleDir).timeout(Duration.ofSeconds(30L)).buildCommand();
        this.commandExecutor.execute(command);
    }

    private void incrementVersion(Path path, Document pom, String nextVersion) {
        Optional<Node> versionNode = this.findVersionNode(pom);
        if (versionNode.isEmpty()) {
            this.logger.warn(ExaError.messageBuilder("W-PK-CORE-196").message("No version element found in pom file {{pom file path}}.", path).mitigation("Please update the version to {{next version}} manually.", nextVersion).toString());
            return;
        }
        if (!this.currentProjectVersion.equals(versionNode.get().getTextContent())) {
            throw new IllegalStateException(ExaError.messageBuilder("E-PK-CORE-174").message("Inconsistent project version {{version in pom file}} found in pom {{pom file path}}, expected {{expected version}}.", versionNode.get().getTextContent(), path, this.currentProjectVersion).ticketMitigation().toString());
        }
        versionNode.get().setTextContent(nextVersion);
    }

    private Optional<Node> findVersionNode(Document pom) {
        Optional<Node> versionNode = this.xmlFileIO.runXPath(pom, "/project/version");
        if (versionNode.isEmpty() || versionNode.get().getTextContent().equals("${revision}")) {
            Optional<Node> revisionElement = this.xmlFileIO.runXPath(pom, "/project/properties/revision");
            this.logger.info("Version element missing or refers to ${revision}, use revision property " + revisionElement);
            return revisionElement;
        }
        return versionNode;
    }

    static String getIncrementedVersion(String version) {
        Semver current = new Semver(version);
        return current.nextPatch().toString();
    }

    private Path getPomPath() {
        VersionConfig versionConfig = this.config.getVersionConfig();
        if (versionConfig instanceof VersionFromSource) {
            VersionFromSource versionFromSource = (VersionFromSource)versionConfig;
            return versionFromSource.getPathToPom();
        }
        return this.projectDir.resolve("pom.xml");
    }
}

