/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.operator.scalar;

import com.facebook.presto.operator.Description;
import com.facebook.presto.operator.scalar.ScalarFunction;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.block.Block;
import com.facebook.presto.spi.block.BlockBuilder;
import com.facebook.presto.spi.block.BlockBuilderStatus;
import com.facebook.presto.spi.type.VarcharType;
import com.facebook.presto.type.LiteralParameters;
import com.facebook.presto.type.SqlType;
import com.facebook.presto.util.Failures;
import com.google.common.primitives.Ints;
import io.airlift.slice.InvalidCodePointException;
import io.airlift.slice.InvalidUtf8Exception;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceUtf8;
import io.airlift.slice.Slices;
import java.text.Normalizer;
import java.util.OptionalInt;
import javax.annotation.Nullable;

public final class StringFunctions {
    private StringFunctions() {
    }

    @Description(value="convert Unicode code point to a string")
    @ScalarFunction
    @SqlType(value="varchar(1)")
    public static Slice chr(@SqlType(value="bigint") long codepoint) {
        try {
            return SliceUtf8.codePointToUtf8((int)Ints.saturatedCast((long)codepoint));
        }
        catch (InvalidCodePointException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Not a valid Unicode code point: " + codepoint, (Throwable)e);
        }
    }

    @Description(value="count of code points of the given string")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="bigint")
    public static long length(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.countCodePoints((Slice)slice);
    }

    @Description(value="greedily removes occurrences of a pattern in a string")
    @ScalarFunction
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="varchar")
    public static Slice replace(@SqlType(value="varchar(x)") Slice str, @SqlType(value="varchar(y)") Slice search) {
        return StringFunctions.replace(str, search, Slices.EMPTY_SLICE);
    }

    @Description(value="greedily replaces occurrences of a pattern with a string")
    @ScalarFunction
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="varchar")
    public static Slice replace(@SqlType(value="varchar(x)") Slice str, @SqlType(value="varchar(y)") Slice search, @SqlType(value="varchar") Slice replace) {
        if (search.length() == 0) {
            int codePointLength;
            Slice buffer = Slices.allocate((int)((SliceUtf8.countCodePoints((Slice)str) + 1) * replace.length() + str.length()));
            buffer.setBytes(0, replace);
            int indexBuffer = replace.length();
            for (int index = 0; index < str.length(); index += codePointLength) {
                codePointLength = SliceUtf8.lengthOfCodePointSafe((Slice)str, (int)index);
                buffer.setBytes(indexBuffer, str, index, codePointLength);
                buffer.setBytes(indexBuffer += codePointLength, replace);
                indexBuffer += replace.length();
            }
            return buffer;
        }
        Slice buffer = Slices.allocate((int)str.length());
        int index = 0;
        int indexBuffer = 0;
        while (index < str.length()) {
            int bytesToCopy;
            int matchIndex = str.indexOf(search, index);
            if (matchIndex < 0) {
                bytesToCopy = str.length() - index;
                buffer = Slices.ensureSize((Slice)buffer, (int)(indexBuffer + bytesToCopy));
                buffer.setBytes(indexBuffer, str, index, bytesToCopy);
                indexBuffer += bytesToCopy;
                break;
            }
            bytesToCopy = matchIndex - index;
            buffer = Slices.ensureSize((Slice)buffer, (int)(indexBuffer + bytesToCopy + replace.length()));
            if (bytesToCopy > 0) {
                buffer.setBytes(indexBuffer, str, index, bytesToCopy);
                indexBuffer += bytesToCopy;
            }
            if (replace.length() > 0) {
                buffer.setBytes(indexBuffer, replace);
                indexBuffer += replace.length();
            }
            index = matchIndex + search.length();
        }
        return buffer.slice(0, indexBuffer);
    }

    @Description(value="reverse all code points in a given string")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice reverse(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.reverse((Slice)slice);
    }

    @Description(value="returns index of first occurrence of a substring (or 0 if not found)")
    @ScalarFunction(value="strpos")
    @SqlType(value="bigint")
    public static long stringPosition(@SqlType(value="varchar") Slice string, @SqlType(value="varchar") Slice substring) {
        if (substring.length() == 0) {
            return 1L;
        }
        int index = string.indexOf(substring);
        if (index < 0) {
            return 0L;
        }
        return SliceUtf8.countCodePoints((Slice)string, (int)0, (int)index) + 1;
    }

    @Description(value="suffix starting at given index")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice substr(@SqlType(value="varchar(x)") Slice utf8, @SqlType(value="bigint") long start) {
        if (start == 0L || utf8.length() == 0) {
            return Slices.EMPTY_SLICE;
        }
        int startCodePoint = Ints.saturatedCast((long)start);
        if (startCodePoint > 0) {
            int indexStart = SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)(startCodePoint - 1));
            if (indexStart < 0) {
                return Slices.EMPTY_SLICE;
            }
            int indexEnd = utf8.length();
            return utf8.slice(indexStart, indexEnd - indexStart);
        }
        int codePoints = SliceUtf8.countCodePoints((Slice)utf8);
        if ((startCodePoint += codePoints) < 0) {
            return Slices.EMPTY_SLICE;
        }
        int indexStart = SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)startCodePoint);
        int indexEnd = utf8.length();
        return utf8.slice(indexStart, indexEnd - indexStart);
    }

    @Description(value="substring of given length starting at an index")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice substr(@SqlType(value="varchar(x)") Slice utf8, @SqlType(value="bigint") long start, @SqlType(value="bigint") long length) {
        if (start == 0L || length <= 0L || utf8.length() == 0) {
            return Slices.EMPTY_SLICE;
        }
        int startCodePoint = Ints.saturatedCast((long)start);
        int lengthCodePoints = Ints.saturatedCast((long)length);
        if (startCodePoint > 0) {
            int indexStart = SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)(startCodePoint - 1));
            if (indexStart < 0) {
                return Slices.EMPTY_SLICE;
            }
            int indexEnd = SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)indexStart, (int)lengthCodePoints);
            if (indexEnd < 0) {
                indexEnd = utf8.length();
            }
            return utf8.slice(indexStart, indexEnd - indexStart);
        }
        int codePoints = SliceUtf8.countCodePoints((Slice)utf8);
        if ((startCodePoint += codePoints) < 0) {
            return Slices.EMPTY_SLICE;
        }
        int indexStart = SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)startCodePoint);
        int indexEnd = startCodePoint + lengthCodePoints < codePoints ? SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)indexStart, (int)lengthCodePoints) : utf8.length();
        return utf8.slice(indexStart, indexEnd - indexStart);
    }

    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="array(varchar(x))")
    public static Block split(@SqlType(value="varchar(x)") Slice string, @SqlType(value="varchar") Slice delimiter) {
        return StringFunctions.split(string, delimiter, string.length() + 1);
    }

    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="array(varchar(x))")
    public static Block split(@SqlType(value="varchar(x)") Slice string, @SqlType(value="varchar") Slice delimiter, @SqlType(value="bigint") long limit) {
        int splitIndex;
        Failures.checkCondition(limit > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Limit must be positive", new Object[0]);
        Failures.checkCondition(limit <= Integer.MAX_VALUE, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Limit is too large", new Object[0]);
        Failures.checkCondition(delimiter.length() > 0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "The delimiter may not be the empty string", new Object[0]);
        BlockBuilder parts = VarcharType.VARCHAR.createBlockBuilder(new BlockBuilderStatus(), 1, string.length());
        if (limit == 1L) {
            VarcharType.VARCHAR.writeSlice(parts, string);
            return parts.build();
        }
        int index = 0;
        while (index < string.length() && (splitIndex = string.indexOf(delimiter, index)) >= 0) {
            VarcharType.VARCHAR.writeSlice(parts, string, index, splitIndex - index);
            index = splitIndex + delimiter.length();
            if ((long)parts.getPositionCount() != limit - 1L) continue;
            break;
        }
        VarcharType.VARCHAR.writeSlice(parts, string, index, string.length() - index);
        return parts.build();
    }

    @Nullable
    @Description(value="splits a string by a delimiter and returns the specified field (counting from one)")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice splitPart(@SqlType(value="varchar(x)") Slice string, @SqlType(value="varchar") Slice delimiter, @SqlType(value="bigint") long index) {
        int matchIndex;
        Failures.checkCondition(index > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Index must be greater than zero", new Object[0]);
        if (delimiter.length() == 0) {
            int startCodePoint = Ints.checkedCast((long)index);
            int indexStart = SliceUtf8.offsetOfCodePoint((Slice)string, (int)(startCodePoint - 1));
            if (indexStart < 0) {
                return null;
            }
            int length = SliceUtf8.lengthOfCodePoint((Slice)string, (int)indexStart);
            if (indexStart + length > string.length()) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid UTF-8 encoding");
            }
            return string.slice(indexStart, length);
        }
        int matchCount = 0;
        int previousIndex = 0;
        while (previousIndex < string.length() && (matchIndex = string.indexOf(delimiter, previousIndex)) >= 0) {
            if ((long)(++matchCount) == index) {
                return string.slice(previousIndex, matchIndex - previousIndex);
            }
            previousIndex = matchIndex + delimiter.length();
        }
        if ((long)matchCount == index - 1L) {
            return string.slice(previousIndex, string.length() - previousIndex);
        }
        return null;
    }

    @Description(value="removes whitespace from the beginning of a string")
    @ScalarFunction(value="ltrim")
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice leftTrim(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.leftTrim((Slice)slice);
    }

    @Description(value="removes whitespace from the end of a string")
    @ScalarFunction(value="rtrim")
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice rightTrim(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.rightTrim((Slice)slice);
    }

    @Description(value="removes whitespace from the beginning and end of a string")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice trim(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.trim((Slice)slice);
    }

    @Description(value="converts the string to lower case")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice lower(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.toLowerCase((Slice)slice);
    }

    @Description(value="converts the string to upper case")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice upper(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.toUpperCase((Slice)slice);
    }

    private static Slice pad(Slice text, long targetLength, Slice padString, int paddingOffset) {
        Failures.checkCondition(0L <= targetLength && targetLength <= Integer.MAX_VALUE, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Target length must be in the range [0..2147483647]", new Object[0]);
        Failures.checkCondition(padString.length() > 0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Padding string must not be empty", new Object[0]);
        int textLength = SliceUtf8.countCodePoints((Slice)text);
        int resultLength = (int)targetLength;
        if (textLength == resultLength) {
            return text;
        }
        if (textLength > resultLength) {
            return SliceUtf8.substring((Slice)text, (int)0, (int)resultLength);
        }
        int padStringLength = SliceUtf8.countCodePoints((Slice)padString);
        int[] padStringCounts = new int[padStringLength];
        for (int i = 0; i < padStringLength; ++i) {
            padStringCounts[i] = SliceUtf8.lengthOfCodePointSafe((Slice)padString, (int)SliceUtf8.offsetOfCodePoint((Slice)padString, (int)i));
        }
        int bufferSize = text.length();
        for (int i = 0; i < resultLength - textLength; ++i) {
            bufferSize += padStringCounts[i % padStringLength];
        }
        Slice buffer = Slices.allocate((int)bufferSize);
        int countBytes = bufferSize - text.length();
        int startPointOfExistingText = (paddingOffset + countBytes) % bufferSize;
        buffer.setBytes(startPointOfExistingText, text);
        int byteIndex = paddingOffset;
        for (int i = 0; i < countBytes / padString.length(); ++i) {
            buffer.setBytes(byteIndex, padString);
            byteIndex += padString.length();
        }
        buffer.setBytes(byteIndex, padString.getBytes(0, paddingOffset + countBytes - byteIndex));
        return buffer;
    }

    @Description(value="pads a string on the left")
    @ScalarFunction(value="lpad")
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="varchar")
    public static Slice leftPad(@SqlType(value="varchar(x)") Slice text, @SqlType(value="bigint") long targetLength, @SqlType(value="varchar(y)") Slice padString) {
        return StringFunctions.pad(text, targetLength, padString, 0);
    }

    @Description(value="pads a string on the right")
    @ScalarFunction(value="rpad")
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="varchar")
    public static Slice rightPad(@SqlType(value="varchar(x)") Slice text, @SqlType(value="bigint") long targetLength, @SqlType(value="varchar(y)") Slice padString) {
        return StringFunctions.pad(text, targetLength, padString, text.length());
    }

    @Description(value="transforms the string to normalized form")
    @ScalarFunction
    @SqlType(value="varchar")
    public static Slice normalize(@SqlType(value="varchar") Slice slice, @SqlType(value="varchar") Slice form) {
        Normalizer.Form targetForm;
        try {
            targetForm = Normalizer.Form.valueOf(form.toStringUtf8());
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Normalization form must be one of [NFD, NFC, NFKD, NFKC]");
        }
        return Slices.utf8Slice((String)Normalizer.normalize(slice.toStringUtf8(), targetForm));
    }

    @Description(value="decodes the UTF-8 encoded string")
    @ScalarFunction
    @SqlType(value="varchar")
    public static Slice fromUtf8(@SqlType(value="varbinary") Slice slice) {
        return SliceUtf8.fixInvalidUtf8((Slice)slice);
    }

    @Description(value="decodes the UTF-8 encoded string")
    @ScalarFunction
    @SqlType(value="varchar")
    public static Slice fromUtf8(@SqlType(value="varbinary") Slice slice, @SqlType(value="varchar") Slice replacementCharacter) {
        OptionalInt replacementCodePoint;
        int count = SliceUtf8.countCodePoints((Slice)replacementCharacter);
        if (count > 1) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Replacement character string must empty or a single character");
        }
        if (count == 1) {
            try {
                replacementCodePoint = OptionalInt.of(SliceUtf8.getCodePointAt((Slice)replacementCharacter, (int)0));
            }
            catch (InvalidUtf8Exception e) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid replacement character");
            }
        } else {
            replacementCodePoint = OptionalInt.empty();
        }
        return SliceUtf8.fixInvalidUtf8((Slice)slice, (OptionalInt)replacementCodePoint);
    }

    @Description(value="decodes the UTF-8 encoded string")
    @ScalarFunction
    @SqlType(value="varchar")
    public static Slice fromUtf8(@SqlType(value="varbinary") Slice slice, @SqlType(value="bigint") long replacementCodePoint) {
        if (replacementCodePoint > 0x10FFFFL || Character.getType((int)replacementCodePoint) == 19) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid replacement character");
        }
        return SliceUtf8.fixInvalidUtf8((Slice)slice, (OptionalInt)OptionalInt.of((int)replacementCodePoint));
    }

    @Description(value="encodes the string to UTF-8")
    @ScalarFunction
    @SqlType(value="varbinary")
    public static Slice toUtf8(@SqlType(value="varchar") Slice slice) {
        return slice;
    }
}

