package com.feingto.cloud.cache.provider;

import com.feingto.cloud.cache.ICache;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * 缓存接口默认实现
 *
 * @author longfei
 */
@SuppressWarnings("unchecked")
public class DefaultCacheProvider implements ICache {
    protected Map<String, Object> data = new HashMap<>();
    private Map<String, Long> expires = new HashMap<>();
    private Map<String, Long> lasts = new HashMap<>();

    protected Object filterExpire(Object obj, String key) {
        return Optional.ofNullable(obj)
                .filter(v -> Objects.nonNull(expires.get(key)))
                .filter(v -> Objects.nonNull(lasts.get(key)))
                .filter(v -> expires.get(key) > 0)
                .filter(v -> System.currentTimeMillis() - lasts.get(key) > expires.get(key))
                .map(v -> {
                    v = null;
                    remove(key);
                    return v;
                })
                .orElse(obj);
    }

    @Override
    public <T> Map<String, T> get() {
        return (Map<String, T>) data;
    }

    @Override
    public <T> T get(String key) {
        return (T) this.filterExpire(data.get(key), key);
    }

    @Override
    public void put(String key, Object value) {
        this.put(key, value, -1);
    }

    @Override
    public void put(String key, Object value, long expireTime) {
        data.put(key, value);
        expires.put(key, expireTime);
        lasts.put(key, System.currentTimeMillis());
    }

    @Override
    public void put(String key, Object value, long expireTime, TimeUnit timeUnit) {
        switch (timeUnit) {
            case DAYS:
                expireTime *= 1000 * 1000 * 1000 * 1000L;
                break;
            case HOURS:
                expireTime *= 1000 * 1000 * 1000L;
                break;
            case MINUTES:
                expireTime *= 1000 * 1000L;
                break;
            case SECONDS:
                expireTime *= 1000L;
                break;
            case MILLISECONDS:
                break;
            case MICROSECONDS:
                expireTime *= 1 / 1000L;
                break;
            case NANOSECONDS:
                expireTime *= 1 / 1000 / 1000L;
                break;
        }
        this.put(key, value, expireTime);
    }

    @Override
    public Set<String> keys() {
        return data.keySet();
    }

    @Override
    public boolean has(String key) {
        return data.containsKey(key);
    }

    @Override
    public void remove(String key) {
        data.remove(key);
        expires.remove(key);
        lasts.remove(key);
    }

    @Override
    public void removeByPrefix(String prefix) {
        this.keys().stream()
                .filter(key -> key.startsWith(prefix))
                .forEach(this::remove);
    }

    @Override
    public void clear() {
        data.clear();
        expires.clear();
        lasts.clear();
    }
}
