package com.feingto.cloud.data.bean;

import lombok.Data;
import lombok.experimental.Accessors;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * 分页模型
 *
 * @author longfei
 */
@Data
@Accessors(chain = true)
public class Page<T> implements Serializable {
    private static final long serialVersionUID = -4600780441241037273L;
    public static final String ORDER_ASC = "ASC";
    public static final String ORDER_DESC = "DESC";

    /**
     * 当前页
     */
    private int pageNumber = 1;

    /**
     * 每页大小
     */
    private int pageSize = 20;

    /**
     * 排序字段
     */
    private String orderField = "id";

    /**
     * 排序规则
     */
    private String orderDirection = ORDER_DESC;

    /**
     * 总记录数
     */
    private long totalElements;

    /**
     * 总页数
     */
    private int totalPages;

    /**
     * 结果集
     */
    private List<T> content = new ArrayList<>();

    /**
     * 是否上一页
     */
    public boolean hasPrevious() {
        return this.pageNumber != 1 && this.pageNumber != 0;
    }

    /**
     * 是否下一页
     */
    public boolean hasNext() {
        return pageNumber + 1 <= totalPages;
    }

    /**
     * 生成Spring Data JPA Pageable
     *
     * @param page Page
     * @return PageRequest
     */
    public static Pageable createPageable(Page page) {
        if (StringUtils.hasText(page.getOrderField())) {
            if (page.getOrderDirection().equalsIgnoreCase(Page.ORDER_ASC)) {
                return PageRequest.of(page.getPageNumber() - 1, page.getPageSize(), Sort.Direction.ASC, page.getOrderField());
            } else {
                return PageRequest.of(page.getPageNumber() - 1, page.getPageSize(), Sort.Direction.DESC, page.getOrderField());
            }
        }

        return PageRequest.of(page.getPageNumber() - 1, page.getPageSize());
    }

    /**
     * 将Spring Data Page的属性注入Page模型
     *
     * @param page     Page<T>
     * @param dataPage org.springframework.data.domain.Page<T>
     */
    public static <T> void injectPageProperties(Page<T> page, org.springframework.data.domain.Page<T> dataPage) {
        page.setTotalElements(dataPage.getTotalElements())
                .setTotalPages(dataPage.getTotalPages())
                .setContent(dataPage.getContent());
    }

    /**
     * 根据记录总数计算分页
     */
    public Page<T> get() {
        this.totalPages = (int) this.totalElements / this.pageSize;

        if (this.totalElements % this.pageSize != 0) {
            this.totalPages++;
        }

        if (this.pageNumber > this.totalPages) {
            this.pageNumber = this.totalPages;
        }

        return this;
    }
}
