package com.feingto.cloud.data.elasticsearch;

import com.feingto.cloud.data.bean.Page;
import com.feingto.cloud.kit.reflection.ReflectionKit;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.elasticsearch.index.query.QueryBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.elasticsearch.core.query.SearchQuery;
import org.springframework.data.elasticsearch.repository.ElasticsearchRepository;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.io.Serializable;
import java.util.List;

/**
 * ElasticSearch 通用泛型服务接口实现
 *
 * @author longfei
 */
@Service
@Transactional(readOnly = true, rollbackFor = Exception.class)
public abstract class BaseService<T, ID extends Serializable> implements IBase<T, ID> {
    @Autowired
    private ElasticsearchRepository<T, ID> repository;

    @Override
    public T index(T entity) {
        return repository.index(entity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public T save(T entity) {
        return repository.save(entity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateByProperty(ID id, String propertyName, Object value) {
        Assert.notNull(id, "Property id is required");
        Assert.notNull(propertyName, "Property name is required");
        repository.findById(id)
                .ifPresent(optional -> {
                    ReflectionKit.setFieldValue(optional, propertyName, value);
                    repository.save(optional);
                });
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(ID id) {
        Assert.notNull(id, "Property id is required");
        repository.deleteById(id);
    }

    @Override
    public long count(QueryBuilder queryBuilder) {
        return Iterables.size(repository.search(queryBuilder));
    }

    @Override
    public T findById(ID id) {
        Assert.notNull(id, "Property id is required");
        return repository.findById(id)
                .orElse(null);
    }

    @Override
    public List<T> findAll(QueryBuilder query) {
        return Lists.newArrayList(repository.search(query));
    }

    @Override
    public Page<T> findSimilarByPage(T entity, String[] fields, Page<T> page) {
        Page.injectPageProperties(page, repository.searchSimilar(entity, fields, Page.createPageable(page)));
        return page;
    }

    @Override
    public Page<T> findByPage(QueryBuilder queryBuilder, Page<T> page) {
        Page.injectPageProperties(page, repository.search(queryBuilder, Page.createPageable(page)));
        return page;
    }

    @Override
    public Page<T> findByPage(SearchQuery searchQuery, Page<T> page) {
        searchQuery.setPageable(Page.createPageable(page));
        Page.injectPageProperties(page, repository.search(searchQuery));
        return page;
    }
}
