package com.feingto.cloud.data.elasticsearch;

import com.feingto.cloud.data.bean.Page;
import org.elasticsearch.index.query.QueryBuilder;
import org.springframework.data.elasticsearch.core.query.SearchQuery;

import java.io.Serializable;
import java.util.List;

/**
 * ElasticSearch 通用泛型服务接口
 *
 * @author longfei
 */
public interface IBase<T, ID extends Serializable> {
    /**
     * 索引
     *
     * @param entity POJO
     * @return entity
     */
    T index(T entity);

    /**
     * 新增/修改
     *
     * @param entity POJO
     * @return entity
     */
    T save(T entity);

    /**
     * 根据 ID 修改属性值
     *
     * @param id           ID
     * @param propertyName 属性名
     * @param value        属性值
     */
    void updateByProperty(ID id, String propertyName, Object value);

    /**
     * 根据 ID 删除
     *
     * @param id ID
     */
    void delete(ID id);

    /**
     * 根据条件统计数量
     *
     * @param queryBuilder QueryBuilder
     */
    long count(QueryBuilder queryBuilder);

    /**
     * 根据 ID 查询
     *
     * @param id ID
     * @return entity
     */
    T findById(ID id);

    /**
     * 查询全部
     */
    List<T> findAll(QueryBuilder query);

    /**
     * 分页查找类似
     *
     * @param entity POJO
     * @param fields 关键字数组
     * @param page   Page<T>
     */
    Page<T> findSimilarByPage(T entity, String[] fields, Page<T> page);

    /**
     * 分页查询
     *
     * @param queryBuilder QueryBuilder
     * @param page         Page<T>
     */
    Page<T> findByPage(QueryBuilder queryBuilder, Page<T> page);

    /**
     * 分页查询
     *
     * @param searchQuery SearchQuery
     * @param page        Page<T>
     */
    Page<T> findByPage(SearchQuery searchQuery, Page<T> page);
}
