package com.feingto.cloud.data.jdbc;

import org.springframework.boot.jdbc.DataSourceBuilder;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * 数据库池（保存外部数据库连接）
 *
 * @author longfei
 */
public class DBPool {
    public static final String EXCHANGE_NAME = "LFExchange";
    /**
     * 数据库池
     */
    private static volatile DBPool instance;
    /**
     * 外部数据库链接
     */
    private static Map<String, DataSource> dbMap = new HashMap<>();

    public static DBPool getInstance() {
        if (Objects.isNull(instance)) {
            synchronized (DBPool.class) {
                if (Objects.isNull(instance)) {
                    instance = new DBPool();
                }
            }
        }
        return instance;
    }

    /**
     * 获取数据库链接
     *
     * @param key 数据源名称
     */
    public Connection getConnection(String key) {
        return Optional.ofNullable(dbMap.get(key))
                .map(pool -> {
                    try {
                        return pool.getConnection();
                    } catch (SQLException e) {
                        throw new RuntimeException("SQL 异常: " + e.getMessage());
                    }
                })
                .orElse(null);
    }

    /**
     * 测试配置文件是否可用
     *
     * @param driver   数据库驱动
     * @param url      数据库链接
     * @param username 数据库用户
     * @param password 数据库密码
     */
    public boolean test(String driver, String url, String username, String password) {
        DataSource dataSource = DataSourceBuilder.create()
                .driverClassName(driver)
                .url(url)
                .username(username)
                .password(password)
                .build();
        try {
            String sql = driver.contains("oracle") ? "SELECT 1 FROM DUAL" : "SELECT 1";
            return dataSource.getConnection().prepareStatement(sql).execute();
        } catch (SQLException e) {
            throw new RuntimeException("SQL 异常: " + e.getMessage());
        }
    }

    /**
     * 创建数据源，并存放到池中
     *
     * @param key      数据源名称（不可重复）
     * @param driver   数据库驱动
     * @param url      数据库链接
     * @param username 用户名
     * @param password 密码
     */
    public DBPool create(String key, String driver, String url, String username, String password) {
        if (!dbMap.containsKey(key)) {
            dbMap.put(key, DataSourceBuilder.create()
                    .driverClassName(driver)
                    .url(url)
                    .username(username)
                    .password(password)
                    .build());
        }
        return this;
    }

    public DBPool create(String key, DataSource dataSource) {
        dbMap.put(key, dataSource);
        return this;
    }

    /**
     * 获取数据源
     *
     * @param key 数据源名称
     */
    public DataSource getDataSource(String key) {
        return dbMap.get(key);
    }

    /**
     * 获取用于交换/汇集的数据库
     */
    public DataSource getLFExchangeDataSource() {
        return dbMap.get(EXCHANGE_NAME);
    }

    /**
     * 是否存在数据源
     *
     * @param key 数据源名称
     */
    public boolean has(String key) {
        return dbMap.containsKey(key);
    }

    /**
     * 销毁指定数据源
     *
     * @param key 数据源名称
     */
    public void destroy(String key) {
        dbMap.remove(key);
    }

    /**
     * 销毁所有数据源
     */
    public void destroyAll() {
        dbMap.clear();
    }

    /**
     * 数据源数量
     */
    public int size() {
        return dbMap.size();
    }
}
