package com.feingto.cloud.data.jdbc;

import com.feingto.cloud.data.jdbc.model.Record;
import com.feingto.cloud.data.jdbc.model.Records;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.support.rowset.SqlRowSet;
import org.springframework.jdbc.support.rowset.SqlRowSetMetaData;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * JdbcTemplate 数据库操作工具
 *
 * @author longfei
 */
@Slf4j
public class JdbcTemplateKit {
    private JdbcTemplate jdbcTemplate;

    public JdbcTemplateKit(JdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    /**
     * 输出Record结果集
     *
     * @param rs SqlRowSet
     * @return Record
     */
    private Record getRecord(SqlRowSet rs) {
        Record record = new Record();
        SqlRowSetMetaData metaData = rs.getMetaData();
        for (int i = 1; i <= metaData.getColumnCount(); i++) {
            String label = metaData.getColumnLabel(i);
            record.set(label, rs.getObject(label));
        }
        return record;
    }

    /**
     * 输出MAP结果集
     *
     * @param rs SqlRowSet
     * @return Map
     */
    private Map<String, Object> getMap(SqlRowSet rs) {
        Map<String, Object> map = new HashMap<>();
        SqlRowSetMetaData metaData = rs.getMetaData();
        for (int i = 1; i <= metaData.getColumnCount(); i++) {
            String label = metaData.getColumnLabel(i);
            map.put(label, rs.getObject(label));
        }
        return map;
    }

    /**
     * 获取单行数据
     *
     * @param sql    Sql字符串
     * @param params 参数数组
     * @return Record
     */
    public Record get(String sql, Object... params) {
        log.debug(sql);
        SqlRowSet rs = jdbcTemplate.queryForRowSet(sql, params);
        return rs.next() ? getRecord(rs) : new Record();
    }

    /**
     * 获取单行数据（Map）
     *
     * @param sql    Sql字符串
     * @param params 参数数组
     * @return Map
     */
    public Map<String, Object> getMap(String sql, Object... params) {
        log.debug(sql);
        SqlRowSet rs = jdbcTemplate.queryForRowSet(sql, params);
        return rs.next() ? getMap(rs) : new HashMap<>();
    }

    /**
     * 获取集合数据
     *
     * @param sql    Sql字符串
     * @param params 参数数组
     * @return Records
     */
    public Records list(String sql, Object... params) {
        log.debug(sql);
        Records records = new Records();
        SqlRowSet rs = jdbcTemplate.queryForRowSet(sql, params);
        while (rs.next()) {
            records.add(getRecord(rs));
        }
        return records;
    }

    /**
     * 获取集合数据
     *
     * @param sql    Sql字符串
     * @param params 参数数组
     * @return List
     */
    public List<Map<String, Object>> listToMap(String sql, Object... params) {
        log.debug(sql);
        List<Map<String, Object>> list = new ArrayList<>();
        SqlRowSet rs = jdbcTemplate.queryForRowSet(sql, params);
        while (rs.next()) {
            list.add(getMap(rs));
        }
        return list;
    }

    /**
     * 执行更新
     *
     * @param sql    Sql字符串
     * @param params 参数数组
     * @return int
     */
    public int update(String sql, Object... params) {
        log.debug(sql);
        return jdbcTemplate.update(sql, params);
    }

    /**
     * 执行
     *
     * @param sql Sql字符串
     */
    public void execute(String sql) {
        log.debug(sql);
        jdbcTemplate.execute(sql);
    }
}
