package com.feingto.cloud.data.jpa;

import com.feingto.cloud.data.bean.OrderSort;
import com.feingto.cloud.data.bean.Page;
import com.feingto.cloud.data.jpa.entity.IdEntity;
import com.feingto.cloud.data.jpa.provider.SqlService;
import com.feingto.cloud.data.jpa.specification.DynamicSpecifications;
import com.feingto.cloud.data.jpa.specification.bean.Condition;
import com.feingto.cloud.data.jpa.specification.bean.Rule;
import com.feingto.cloud.kit.reflection.BeanConvertKit;
import com.feingto.cloud.kit.reflection.ReflectionKit;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.persistence.criteria.JoinType;
import java.io.Serializable;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * 通用泛型服务接口实现
 *
 * @author longfei
 */
@Service
@Transactional(readOnly = true, rollbackFor = Exception.class)
public abstract class BaseService<T, ID extends Serializable> extends SqlService<T> implements IBase<T, ID> {
    private static final String PARENT_ID = "-1";

    @Autowired
    private JpaRepository<T, ID> repository;

    @Autowired
    private JpaSpecificationExecutor<T> specificationExecutor;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public T save(T entity) {
        return super.resloveLazyInit(repository.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public T update(ID id, T entity) {
        T _entity = repository.getOne(id);
        BeanConvertKit.copyProperties(entity, _entity);
        return super.resloveLazyInit(repository.save(_entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateByProperty(ID id, String propertyName, Object value) {
        Assert.notNull(id, "Property id is required");
        Assert.notNull(propertyName, "Property name is required");
        repository.findById(id)
                .ifPresent(optional -> {
                    ReflectionKit.setFieldValue(optional, propertyName, value);
                    repository.save(optional);
                });
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateByProperty(ID id, Condition value) {
        Assert.notNull(id, "Property id is required");
        repository.findById(id)
                .ifPresent(optional -> repository.save(this.filter(optional, value.getRules())));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateByProperty(Condition condition, Condition value) {
        Assert.notNull(condition, "Property id is required");
        Optional.of(this.findAll(condition))
                .ifPresent(list -> list.forEach(optional ->
                        repository.save(this.filter(optional, value.getRules()))));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(ID id) {
        Assert.notNull(id, "Property id is required");
        repository.deleteById(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(ID[] ids) {
        Stream.of(ids).forEach(this::delete);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(Condition condition) {
        Optional.of(this.findAll(condition))
                .ifPresent(list -> repository.deleteAll(list));
    }

    @Override
    public Long count() {
        return repository.count();
    }

    @Override
    public Long count(Condition condition) {
        Assert.notNull(condition, "Condition is required");
        Specification<T> specification = DynamicSpecifications.byCondition(condition);
        return specificationExecutor.count(specification);
    }

    @Override
    public T findById(ID id) {
        Assert.notNull(id, "Property id is required");
        return repository.findById(id)
                .map(super::resloveLazyInit)
                .orElse(null);
    }

    @Override
    public T findOne(Condition condition) {
        Assert.notNull(condition, "Condition is required");
        Specification<T> specification = DynamicSpecifications.byCondition(condition);
        return specificationExecutor.findOne(specification)
                .map(super::resloveLazyInit)
                .orElse(null);
    }

    @Override
    public List<T> findAll() {
        return super.resloveLazyInit(repository.findAll());
    }

    @Override
    public List<T> findAll(OrderSort sort) {
        return super.resloveLazyInit(repository.findAll(this.createSort(sort)));
    }

    @Override
    public List<T> findAll(Condition condition) {
        return this.findAll(condition, null);
    }

    @Override
    public List<T> findAll(Condition condition, OrderSort sort) {
        return this.findAll(condition, sort, JoinType.LEFT);
    }

    @Override
    public List<T> findAll(Condition condition, OrderSort sort, JoinType joinType) {
        Specification<T> specification = DynamicSpecifications.byCondition(condition, joinType);
        return super.resloveLazyInit(specificationExecutor.findAll(specification, this.createSort(sort)));
    }

    @Override
    public List<T> findAllByPage(Condition condition) {
        List<T> result = new ArrayList<>();
        int pageNumber = 1;
        boolean haveNext;

        do {
            Page<T> page = this.findByPage(condition, new Page<T>()
                    .setPageNumber(pageNumber)
                    .setPageSize(50));
            result.addAll(page.getContent());
            haveNext = page.hasNext();
            pageNumber++;
        } while (haveNext);

        return result;
    }

    @Override
    public Page<T> findByPage(Page<T> page) {
        org.springframework.data.domain.Page<T> springDataPage = repository.findAll(Page.createPageable(page));
        super.resloveLazyInit(springDataPage.getContent());
        Page.injectPageProperties(page, springDataPage);
        return page;
    }

    @Override
    public Page<T> findByPage(Condition condition, Page<T> page) {
        return this.findByPage(condition, page, JoinType.LEFT);
    }

    @Override
    public Page<T> findByPage(Condition condition, Page<T> page, JoinType joinType) {
        Specification<T> specification = DynamicSpecifications.byCondition(condition, joinType);
        org.springframework.data.domain.Page<T> springDataPage = specificationExecutor.findAll(specification,
                Page.createPageable(page));
        super.resloveLazyInit(springDataPage.getContent());
        Page.injectPageProperties(page, springDataPage);
        return page;
    }

    private T filter(T t, List<Rule> rules) {
        rules.stream()
                .filter(rule -> rule.getOp().equals(Rule.Operator.EQ))
                .forEach(rule -> ReflectionKit.setFieldValue(t, rule.getProperty(), rule.getValue()));
        return t;
    }

    private Sort createSort(OrderSort orderSort) {
        Sort sort;
        if (Objects.isNull(orderSort)) {
            sort = new Sort(Sort.Direction.DESC, "id");
        } else if (orderSort.getOrderDirection().equalsIgnoreCase(Page.ORDER_ASC)) {
            sort = new Sort(Sort.Direction.ASC, orderSort.getOrderField());
        } else {
            sort = new Sort(Sort.Direction.DESC, orderSort.getOrderField());
        }
        return sort;
    }

    /**
     * 根据满足条件的数量是否为0校验重复性
     *
     * @param entity    父类为IdEntity的实体
     * @param condition 条件表达式
     * @param error     错误提示
     */
    protected void checkRepeat(IdEntity entity, Condition condition, String error) {
        if (entity.isNew()) {
            Assert.state(this.count(condition) == 0, error);
        } else {
            Assert.state(this.count(condition.ne("id", entity.getId())) == 0, error);
        }
    }

    /**
     * 生成树结构数据的编码 code
     * 规则 001, 001001, 001002
     *
     * @param parentId 父ID
     * @return code
     */
    protected String createCode(ID parentId) {
        String code;
        String maxCode = "000";
        Object obj = super.find("select code from " + super.clazz.getSimpleName() + " where parentId='" + parentId + "' order by code desc");
        if (Objects.nonNull(obj) && ((List) obj).size() > 0) {
            maxCode = ((List) obj).get(0).toString();
        }
        if (PARENT_ID.equals(parentId)) {
            int count = Integer.parseInt(maxCode);
            ++count;
            DecimalFormat df = new DecimalFormat("000");
            code = df.format((long) count);
        } else {
            int count = Integer.parseInt(maxCode.substring(maxCode.length() - 3));
            T t = repository.findById(parentId).orElse(null);
            code = Objects.nonNull(t) ? (String) ReflectionKit.getFieldValue(t, "code") : "";
            ++count;
            DecimalFormat df = new DecimalFormat("000");
            code = code + df.format((long) count);
        }
        return code;
    }
}
