package com.feingto.cloud.data.jpa.generator;

import com.feingto.cloud.data.jpa.generator.model.Entity;
import com.feingto.cloud.data.jpa.generator.model.Property;
import com.feingto.cloud.data.jpa.utils.PersistenceUtils;
import com.feingto.cloud.kit.DateKit;
import com.google.common.collect.Maps;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import lombok.Setter;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 自动生成实体类对应的接口类和服务类
 * 1. 实体类必须有@Id注解，没有主键的实体无效
 * 2. 加入依赖 spring-boot-starter-freemarker
 * main方法: new Generator().target("feingto-account/src/main/java").run(User.class);
 *
 * @author longfei
 */
@Slf4j
@Accessors(fluent = true)
public class Generator {
    private static Configuration configuration;

    /**
     * 目标目录
     */
    @Setter
    private String target;

    /**
     * 文件存在时是否覆盖旧文件
     */
    @Setter
    private boolean override = false;

    public Generator() {
        configuration = new Configuration(Configuration.VERSION_2_3_28);
        configuration.setDefaultEncoding(StandardCharsets.UTF_8.name());
        configuration.setBooleanFormat("true, false");
        configuration.setDateTimeFormat(DateKit.DATE_TIME);
        configuration.setDateFormat(DateKit.DATE);
        configuration.setTimeFormat(DateKit.TIME);
        configuration.setTemplateUpdateDelayMilliseconds(0);
        configuration.setClassForTemplateLoading(Generator.class, "/templates/");
        configuration.unsetCacheStorage();
    }

    /**
     * 自动生成实体类对应的接口类和服务类
     *
     * @param clazz 实体类
     */
    public void run(Class<?> clazz) {
        makeRepository(clazz);
        makeSerivce(clazz);
    }

    /**
     * 生成 *Repository.java 文件
     */
    private void makeRepository(Class<?> clazz) {
        List<Property> properties = new ArrayList<>();
        Property pkProperty = PersistenceUtils.getPkProperty(clazz);
        if (StringUtils.hasText(pkProperty.getPropertyName())) {
            properties.add(pkProperty);
            Entity entity = new Entity()
                    .setJavaPackage(clazz.getPackage().getName())
                    .setClassName(clazz.getSimpleName())
                    .setSuperclass("MyRepository")
                    .setSuperPackage("com.feingto.cloud.orm.jpa.repository")
                    .setProperties(properties)
                    .setTargetPackage(clazz.getPackage().getName().replace("domain", "repository"));
            try {
                Map<String, Object> model = Maps.newHashMap();
                model.put("entity", entity);
                Template template = configuration.getTemplate("jpa/repository.ftl");

                File outDir = new File(target);
                if (!outDir.exists()) {
                    outDir.mkdir();
                }

                File javaFile = PersistenceUtils.createJava(outDir, entity.getTargetPackage(), entity.getClassName() + "Repository");
                if (!javaFile.exists() || override) {
                    Writer javaWriter = new FileWriter(javaFile);
                    template.process(model, javaWriter);
                    log.debug("文件生成路径： {}", javaFile.getCanonicalPath());
                    javaWriter.flush();
                    javaWriter.close();
                }
            } catch (IOException | TemplateException e) {
                throw new RuntimeException("Generator code exception occured：" + e.getMessage());
            }
        }
    }

    /**
     * 生成 *Service.java 文件
     */
    private void makeSerivce(Class<?> clazz) {
        List<Property> properties = new ArrayList<>();
        Property pkProperty = PersistenceUtils.getPkProperty(clazz);
        if (StringUtils.hasText(pkProperty.getPropertyName())) {
            properties.add(pkProperty);
            Entity entity = new Entity()
                    .setJavaPackage(clazz.getPackage().getName())
                    .setClassName(clazz.getSimpleName())
                    .setSuperclass("BaseService")
                    .setSuperPackage("com.feingto.cloud.orm.jpa")
                    .setProperties(properties)
                    .setTargetPackage(clazz.getPackage().getName().replace("domain", "service") + ".impl");
            try {
                Map<String, Object> model = Maps.newHashMap();
                model.put("entity", entity);
                Template template = configuration.getTemplate("jpa/service.ftl");

                File outDir = new File(target);
                if (!outDir.exists()) {
                    outDir.mkdir();
                }

                File javaFile = PersistenceUtils.createJava(outDir, entity.getTargetPackage(), entity.getClassName() + "Service");
                if (!javaFile.exists() || override) {
                    Writer javaWriter = new FileWriter(javaFile);
                    template.process(model, javaWriter);
                    log.debug("文件生成路径： {}", javaFile.getCanonicalPath());
                    javaWriter.flush();
                    javaWriter.close();
                }
            } catch (IOException | TemplateException e) {
                throw new RuntimeException("Generator code exception occured：" + e.getMessage());
            }
        }
    }
}
