package com.feingto.cloud.data.jpa.provider;

import com.feingto.cloud.kit.reflection.ReflectionKit;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import java.util.List;
import java.util.Objects;

/**
 * 底层sql、hql操作接口实现类
 *
 * @author longfei
 */
@Service
@Transactional(readOnly = true, rollbackFor = Exception.class)
@SuppressWarnings("unchecked")
public abstract class SqlService<T> implements ISqlProvider<T> {
    protected final Class<T> clazz;

    private final ThreadLocal<ILazyInitializer<T>> lazyTreadLocal = new ThreadLocal<>();

    @PersistenceContext
    protected EntityManager em;

    public SqlService() {
        this.clazz = ReflectionKit.getClassGenricType(getClass());
    }

    @Override
    public void setLazyInitializer(ILazyInitializer<T> lazyInit) {
        this.lazyTreadLocal.set(lazyInit);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void execute(String hql, Object... params) {
        Query query = em.createQuery(hql);
        this.setParameters(query, params);
        query.executeUpdate();
    }

    @Override
    public long count(String hql, Object... params) {
        Query query = em.createQuery(hql);
        this.setParameters(query, params);
        Object obj = query.getSingleResult();
        return Objects.nonNull(obj) ? Long.parseLong(obj.toString()) : 0;
    }

    @Override
    public Object getSingle(String hql, Object... params) {
        Query query = em.createQuery(hql);
        this.setParameters(query, params);
        return query.getSingleResult();
    }

    @Override
    public List find(String hql, Object... params) {
        Query query = em.createQuery(hql);
        this.setParameters(query, params);
        return query.getResultList();
    }

    @Override
    public T findOne(String hql, Object... params) {
        Query query = em.createQuery(hql);
        this.setParameters(query, params);
        return this.resloveLazyInit((T) query.getSingleResult());
    }

    @Override
    public List<T> findAll(String hql, Object... params) {
        Query query = em.createQuery(hql);
        this.setParameters(query, params);
        return this.resloveLazyInit((List<T>) query.getResultList());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void executeBySql(String sql, Object... params) {
        Query query = em.createNativeQuery(sql);
        this.setParameters(query, params);
        query.executeUpdate();
    }

    @Override
    public long countBySql(String sql, Object... params) {
        Query query = em.createNativeQuery(sql);
        this.setParameters(query, params);
        Object obj = query.getSingleResult();
        return Objects.nonNull(obj) ? Long.parseLong(obj.toString()) : 0;
    }

    @Override
    public Object getSingleBySql(String sql, Object... params) {
        Query query = em.createNativeQuery(sql);
        this.setParameters(query, params);
        return query.getSingleResult();
    }

    @Override
    public List findBySql(String sql, Object... params) {
        Query query = em.createNativeQuery(sql);
        this.setParameters(query, params);
        return query.getResultList();
    }

    @Override
    public T findOneBySql(String sql, Object... params) {
        Query query = em.createNativeQuery(sql, clazz);
        this.setParameters(query, params);
        return this.resloveLazyInit((T) query.getSingleResult());
    }

    @Override
    public List<T> findAllBySql(String sql, Object... params) {
        Query query = em.createNativeQuery(sql, clazz);
        this.setParameters(query, params);
        return this.resloveLazyInit((List<T>) query.getResultList());
    }

    private void setParameters(Query query, Object... params) {
        for (int i = 0; i < params.length; i++) {
            query.setParameter(i + 1, params[i]);
        }
    }

    public T resloveLazyInit(T t) {
        if (Objects.nonNull(this.lazyTreadLocal.get()) && Objects.nonNull(t)) {
            this.lazyTreadLocal.get().init(t);
        }
        this.lazyTreadLocal.remove();
        return t;
    }

    protected List<T> resloveLazyInit(List<T> list) {
        if (Objects.nonNull(this.lazyTreadLocal.get()) && CollectionUtils.isNotEmpty(list)) {
            list.forEach(t -> this.lazyTreadLocal.get().init(t));
        }
        this.lazyTreadLocal.remove();
        return list;
    }
}
