package com.feingto.cloud.data.jpa.serializer;

import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.hibernate.Hibernate;
import org.hibernate.collection.spi.PersistentCollection;
import org.hibernate.proxy.HibernateProxy;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.util.StringUtils;

import java.lang.reflect.Modifier;
import java.util.*;

/**
 * 用于Hibernate Object 的序列化
 * 访问延迟加载的属性返回null值
 * <p>
 * Role proxyRole = (Role) new HibernateBeanSerializer(role).getProxy();
 * assertNull(proxyRole.getResource());// 延迟加载, 为null
 * Hibernate.initialize(role.getResource());
 * assertNotNull(proxyRole.getResource());// 不为null
 *
 * @param <T>
 * @author longfei
 */
@SuppressWarnings("unchecked")
public class HibernateBeanSerializer<T> {
    private T proxy;

    public HibernateBeanSerializer(T obj, String... excludesProperties) {
        if (Objects.isNull(obj)) {
            this.proxy = null;
        } else {
            ProxyFactory pf = new ProxyFactory();
            pf.setTargetClass(obj.getClass());
            pf.setOptimize(true);
            pf.setTarget(obj);
            pf.setProxyTargetClass(true);
            pf.setOpaque(true);
            pf.setExposeProxy(true);
            pf.setPreFiltered(true);
            HibernateBeanSerializerAdvice beanSerializerAdvice = new HibernateBeanSerializerAdvice();
            beanSerializerAdvice.setExcludesProperties(excludesProperties);
            pf.addAdvice(beanSerializerAdvice);
            this.proxy = (T) pf.getProxy();
        }
    }

    public T getProxy() {
        return this.proxy;
    }

    static private class HibernateBeanSerializerAdvice implements MethodInterceptor {
        private String[] excludesProperties = new String[0];

        private static String getPropertyName(String methodName) {
            String propertyName = null;
            if (methodName.startsWith("get")) {
                propertyName = methodName.substring("get".length());
            } else if (methodName.startsWith("is")) {
                propertyName = methodName.substring("is".length());
            } else if (methodName.startsWith("set")) {
                propertyName = methodName.substring("set".length());
            }
            return Objects.isNull(propertyName) ? null : StringUtils.uncapitalize(propertyName);
        }

        public String[] getExcludesProperties() {
            return excludesProperties;
        }

        public void setExcludesProperties(String[] excludesProperties) {
            this.excludesProperties = Objects.isNull(excludesProperties) ? new String[0] : excludesProperties;
        }

        public Object invoke(MethodInvocation mi) throws Throwable {
            String propertyName = getPropertyName(mi.getMethod().getName());
            Class returnType = mi.getMethod().getReturnType();
            if (Objects.isNull(propertyName))
                return mi.proceed();
            if (!Hibernate.isPropertyInitialized(mi.getThis(), propertyName))
                return null;
            if (isExclude(propertyName))
                return null;
            Object returnValue = mi.proceed();
            return processReturnValue(returnType, returnValue);
        }

        private Object processReturnValue(Class returnType, Object returnValue) {
            if (Objects.isNull(returnValue))
                return null;
            if (Objects.nonNull(returnType) && Modifier.isFinal(returnType.getModifiers()))
                return returnValue;
            // This might be a lazy-collection so we need to double check
            if (!Hibernate.isInitialized(returnValue))
                return null;
            // This is Hibernate Object
            if (returnValue instanceof HibernateProxy) {
                return new HibernateBeanSerializer(returnValue).getProxy();
            } else if (returnValue instanceof PersistentCollection) {
                if (Objects.isNull(returnType))
                    return null;
                if (returnType.isAssignableFrom(Map.class)) {
                    Map proxyMap = new LinkedHashMap<>();
                    Map map = (Map) returnValue;
                    Set<Map.Entry> entrySet = map.entrySet();
                    for (Map.Entry entry : entrySet) {
                        proxyMap.put(entry.getKey(), new HibernateBeanSerializer(entry.getValue()));
                    }
                    return proxyMap;
                }
                Collection proxyCollection;
                if (returnType.isAssignableFrom(Set.class)) {
                    proxyCollection = new LinkedHashSet();
                } else if (returnType.isAssignableFrom(List.class)) {
                    proxyCollection = new ArrayList();
                } else {
                    return returnValue;
                }
                for (Object o : (Collection) returnValue) {
                    proxyCollection.add(new HibernateBeanSerializer(o).getProxy());
                }
                return proxyCollection;
            } else {
                return returnValue;
            }
        }

        private boolean isExclude(String propertyName) {
            for (String excludePropertyName : excludesProperties) {
                if (propertyName.equals(excludePropertyName)) {
                    return true;
                }
            }
            return false;
        }
    }
}
