package com.feingto.cloud.data.jpa.specification;

import com.feingto.cloud.data.jpa.specification.bean.Condition;
import com.feingto.cloud.data.jpa.specification.bean.Rule;
import com.feingto.cloud.data.jpa.specification.bean.RuleGroup;
import com.feingto.cloud.kit.DateKit;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.EnumUtils;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.util.Assert;

import javax.persistence.criteria.*;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 通用表达式查询封装, 支持属性表达式嵌套, 联合运算查询.
 *
 * @author longfei
 */
@Slf4j
public class DynamicSpecifications {
    /**
     * 组装 Specification 规则
     *
     * @param condition 查询参数封装 OperatorRule, Rule,Search
     * @return Specification
     */
    public static <T> Specification<T> byCondition(final Condition condition) {
        return byCondition(condition, JoinType.LEFT);
    }

    /**
     * 组装 Specification 规则
     *
     * @param condition Search
     * @param joinType  JoinType.INNER(Default), JoinType.LEFT, JoinType.RIGHT
     * @return Specification
     */
    public static <T> Specification<T> byCondition(final Condition condition, final JoinType joinType) {
        Assert.notNull(condition, "Parameter condition can't be null");

        return (root, query, builder) -> {
            List<Predicate> predicatesUnion = new ArrayList<>();
            condition.getConditions().forEach(cond -> makeRuleGroup(predicatesUnion, cond, root, builder, joinType));
            makeRuleGroup(predicatesUnion, condition, root, builder, joinType);

            if (CollectionUtils.isNotEmpty(predicatesUnion)) {
                if (predicatesUnion.size() == 1) {
                    applyGroupByAndDistinct(root, query, condition);
                    return predicatesUnion.get(0);
                } else if (predicatesUnion.size() > 1) {
                    RuleGroup.UnionOperator union = Objects.nonNull(condition.getOp()) ? condition.getOp() : RuleGroup.UnionOperator.AND;
                    Predicate predicate = null;
                    switch (union) {
                        case AND:
                            predicate = builder.and(predicatesUnion.toArray(new Predicate[0]));
                            break;
                        case OR:
                            predicate = builder.or(predicatesUnion.toArray(new Predicate[0]));
                            break;
                    }
                    if (Objects.nonNull(predicate)) {
                        applyGroupByAndDistinct(root, query, condition);
                        return predicate;
                    }
                }
            }
            applyGroupByAndDistinct(root, query, condition);
            return builder.conjunction();
        };
    }

    /**
     * 组装 List<Predicate> 规则
     */
    private static <T> void makeRuleGroup(List<Predicate> predicatesUnion, Condition condition, Root<T> root, CriteriaBuilder builder, JoinType joinType) {
        List<Predicate> groupPredicates = new ArrayList<>();
        for (RuleGroup group : condition.getGroups()) {
            if (CollectionUtils.isEmpty(group.getRules())) {
                continue;
            }

            List<Predicate> predicates = new ArrayList<>();
            group.getRules().forEach(rule -> toPredicates(predicates, rule, makePath(rule, root, joinType), builder));
            if (predicates.size() > 0) {
                RuleGroup.UnionOperator union = Objects.nonNull(group.getOp()) ? group.getOp() : RuleGroup.UnionOperator.AND;
                makeUnion(groupPredicates, predicates, union, builder);
            }
        }

        condition.getRules().forEach(rule -> toPredicates(groupPredicates, rule, makePath(rule, root, joinType), builder));
        if (groupPredicates.size() > 0) {
            RuleGroup.UnionOperator union = Objects.nonNull(condition.getOp()) ? condition.getOp() : RuleGroup.UnionOperator.AND;
            makeUnion(predicatesUnion, groupPredicates, union, builder);
        }
    }

    /**
     * 组装 Path
     */
    private static <T> Path makePath(Rule rule, Root<T> root, JoinType joinType) {
        Join join = null;
        String[] names = rule.getProperty().split("\\.");
        if (names.length > 1) {
            join = applyJoin(names, root, joinType);
        }
        return Objects.nonNull(join) ? join.get(names[names.length - 1]) : root.get(names[0]);
    }

    /**
     * 组装 List<Predicate> 规则, 操作符连接
     */
    private static void makeUnion(List<Predicate> predicatesUnion, List<Predicate> predicates, RuleGroup.UnionOperator union, CriteriaBuilder builder) {
        switch (union) {
            case AND:
                predicatesUnion.add(builder.and(predicates.toArray(new Predicate[0])));
                break;
            case OR:
                predicatesUnion.add(builder.or(predicates.toArray(new Predicate[0])));
                break;
        }
    }

    /**
     * 嵌套的路径转换
     * 例如对象 Group 有 user 属性名,表达式为 "user.name"，转换为 Group.user.name 属性
     */
    private static <T> Join applyJoin(String[] names, Root<T> root, JoinType joinType) {
        Join join = root.join(names[0], joinType);
        for (int i = 1; i < names.length - 1; i++) {
            join = join.join(names[i], joinType);
        }
        return join;
    }

    /**
     * 组装 Predicate 集合
     *
     * @param predicates   原来的 Predicate 集合
     * @param operatorRule 运算规则对象 Rule.OperatorRule
     * @param expression   Path
     * @param builder      CriteriaBuilder
     */
    @SuppressWarnings("unchecked")
    private static void toPredicates(List<Predicate> predicates, Rule operatorRule, Path expression, CriteriaBuilder builder) {
        Class clazz = expression.getJavaType();
        Object value = operatorRule.getValue();
        if (Date.class.isAssignableFrom(clazz) && !value.getClass().equals(clazz)) {
            operatorRule.setValue(convert2Date((String) value));
        } else if (Enum.class.isAssignableFrom(clazz) && Objects.nonNull(value) && !value.getClass().equals(clazz)) {
            if (List.class.isAssignableFrom(value.getClass())) {
                operatorRule.setValue(((List) value).stream().map(v -> convert2Enum(clazz, (String) v)));
            } else {
                operatorRule.setValue(convert2Enum(clazz, (String) value));
            }
        }
        switch (operatorRule.getOp()) {
            case EQ:
                predicates.add(builder.equal(expression, value));
                break;
            case NEQ:
                predicates.add(builder.notEqual(expression, value));
                break;
            case SLIKE:
                predicates.add(builder.like(expression, value + "%"));
                break;
            case ELIKE:
                predicates.add(builder.like(expression, "%" + value));
                break;
            case LIKE:
                predicates.add(builder.like(expression, "%" + value + "%"));
                break;
            case GT:
                predicates.add(builder.greaterThan(expression, (Comparable) value));
                break;
            case LT:
                predicates.add(builder.lessThan(expression, (Comparable) value));
                break;
            case GTE:
                predicates.add(builder.greaterThanOrEqualTo(expression, (Comparable) value));
                break;
            case LTE:
                predicates.add(builder.lessThanOrEqualTo(expression, (Comparable) value));
                break;
            case IN:
                predicates.add(builder.and(expression.in(value)));
                break;
            case NIN:
                predicates.add(builder.not((expression.in(value))));
                break;
            case ISEMPTY:
                predicates.add(builder.isEmpty(expression));
                break;
            case ISNOTEMPTY:
                predicates.add(builder.isNotEmpty(expression));
                break;
            case ISNULL:
                predicates.add(builder.isNull(expression));
                break;
            case ISNOTNULL:
                predicates.add(builder.isNotNull(expression));
                break;
            case ISBOOLEAN:
                if ((value instanceof Boolean && (Boolean) value) || (value instanceof String && "true".equals(value))) {
                    predicates.add(builder.isTrue(expression));
                } else {
                    predicates.add(builder.isFalse(expression));
                }
                break;
            case ISTRUE:
                predicates.add(builder.isTrue(expression));
                break;
            case ISFALSE:
                predicates.add(builder.isFalse(expression));
                break;
        }
    }

    /**
     * group by 和 distinct 查询
     */
    private static <T> void applyGroupByAndDistinct(Root<T> root, CriteriaQuery<?> query, Condition condition) {
        List<Expression<?>> expGroupBys = new ArrayList<>();
        condition.getGroupByNames().forEach(name -> {
            Path expGroupBy = null;
            String[] names = name.split("\\.");
            for (int i = 0; i < names.length; i++) {
                expGroupBy = (i == 0 ? root : expGroupBy).get(names[i]);
            }
            expGroupBys.add(expGroupBy);
        });
        if (expGroupBys.size() > 0) {
            query.groupBy(expGroupBys);
        }
        query.distinct(condition.isDistinct());
    }

    private static Date convert2Date(String dateString) {
        SimpleDateFormat sFormat = new SimpleDateFormat(DateKit.DATE);
        try {
            return sFormat.parse(dateString);
        } catch (ParseException e) {
            try {
                return sFormat.parse(DateKit.DATE_TIME);
            } catch (ParseException e1) {
                try {
                    return sFormat.parse(DateKit.TIME);
                } catch (ParseException e2) {
                    log.error("Convert time is error! The dateString is {}.{}", dateString, e2.getMessage());
                }
            }
        }
        return null;
    }

    private static <E extends Enum<E>> E convert2Enum(Class<E> enumClass, String enumString) {
        return EnumUtils.getEnum(enumClass, enumString);
    }
}
