package com.feingto.cloud.data.jpa.specification.bean;

import com.feingto.cloud.kit.reflection.BeanConvertKit;
import lombok.Data;
import lombok.experimental.Accessors;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.util.WebUtils;

import javax.servlet.http.HttpServletRequest;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 查询参数过滤规则
 * <p>
 * JSON格式:
 * {"conditions":[{"groups":[{"rules":[{"property":"id.sn","op":"EQ","value":"001"},{"property":"enabled","op":"EQ","value":"ENABLED"}],"op":"and"}],"rules":[{"property":"type","op":"EQ","value":1}],"op":"or"}],"rules":[{"property":"status","op":"EQ","value":1}],"op":"and"}
 *
 * @author longfei
 */
@Data
@Accessors(chain = true)
public class Condition implements Serializable {
    private static final long serialVersionUID = 4950188917289587920L;

    /**
     * 动态查询，参数前缀
     */
    private static final String SEARCH_PREFIX = "search_";

    private List<Condition> conditions = new ArrayList<>();

    private List<RuleGroup> groups = new ArrayList<>();

    private List<Rule> rules = new ArrayList<>();

    /**
     * 操作符(AND 或者 OR)
     */
    private RuleGroup.UnionOperator op;

    /**
     * group by
     * 此属性不支持 Oracle, Oracle select 出来的字段必须在 group by 后边出现.
     * <p>
     * Oracle group by @Query 使用说明:
     * 返回单属性集合 List<String>: @Query("select e.setProperty from Entity e group by e.setProperty")
     * 返回多属性集合 List<Entity>: @Query("select new Entity(e.property1, e.property2) from Entity e group by e.property1, e.property2"), 必须要有public Entity(String property1, String property2)构造函数.
     */
    private List<String> groupByNames = new ArrayList<>();

    /**
     * 是否去重
     */
    private boolean distinct;

    /**
     * 备注
     */
    private String remark;

    public static Condition build() {
        return new Condition();
    }

    /**
     * request参数转换为查询规则
     * 参数名示例：
     * search_EQ_status
     * search_LIKE_username_OR_realName
     */
    public static Condition build(HttpServletRequest request) {
        Condition condition = Condition.build();
        if (Objects.nonNull(request)) {
            Map<String, Object> searchParams = WebUtils.getParametersStartingWith(request, SEARCH_PREFIX);
            for (String key : searchParams.keySet()) {
                String value = (String) searchParams.get(key);
                if (StringUtils.isBlank(value)) {
                    continue;
                }
                String filedName = StringUtils.substringAfter(key, "_");
                Rule.Operator operator = Rule.Operator.valueOf(StringUtils.substringBefore(key, "_"));
                if (StringUtils.indexOf(filedName, RuleGroup.UnionOperator.OR.getName()) > 0
                        || StringUtils.indexOf(filedName, RuleGroup.UnionOperator.AND.getName()) > 0) {
                    Condition.parseFiledName(condition, filedName, operator, value);
                } else {
                    condition.getRules().add(new Rule()
                            .setProperty(filedName)
                            .setOp(operator)
                            .setValue(value.contains(",") ? Stream.of(value.split(","))
                                    .collect(Collectors.toList()) : value.trim()));
                }
            }
        }
        return condition;
    }

    /**
     * 分析 filedName, 提取 AND/OR 并转换为查询规则
     */
    private static Condition parseFiledName(Condition condition, String filedName, Rule.Operator operator, String value) {
        String[] names = StringUtils.splitByWholeSeparator(filedName, "_");
        if (names.length == 0) {
            return condition;
        }
        condition.and();
        RuleGroup.UnionOperator union = null;
        for (String name : names) {
            if (!RuleGroup.UnionOperator.OR.getName().equals(name) && !RuleGroup.UnionOperator.AND.getName().equals(name)) {
                condition.getRules().add(new Rule()
                        .setProperty(name)
                        .setOp(operator)
                        .setValue(value.contains(",") ? Stream.of(value.split(","))
                                .collect(Collectors.toList()) : value.trim()));

            } else {
                union = RuleGroup.UnionOperator.valueOf(name);
            }
        }
        if (Objects.nonNull(union)) {
            if (RuleGroup.UnionOperator.AND.equals(union)) {
                condition.and();
            } else if (RuleGroup.UnionOperator.OR.equals(union)) {
                condition.or();
            }
        }
        return condition;
    }

    public Condition eq(String property, Object value) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.EQ).setValue(value));
        return this;
    }

    public Condition ne(String property, Object value) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.NEQ).setValue(value));
        return this;
    }

    public Condition slike(String property, Object value) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.SLIKE).setValue(value));
        return this;
    }

    public Condition elike(String property, Object value) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.ELIKE).setValue(value));
        return this;
    }

    public Condition like(String property, Object value) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.LIKE).setValue(value));
        return this;
    }

    public Condition gt(String property, Object value) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.GT).setValue(value));
        return this;
    }

    public Condition lt(String property, Object value) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.LT).setValue(value));
        return this;
    }

    public Condition gte(String property, Object value) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.GTE).setValue(value));
        return this;
    }

    public Condition lte(String property, Object value) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.LTE).setValue(value));
        return this;
    }

    public Condition between(String property, Object lo, Object hi) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.GT).setValue(lo));
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.LT).setValue(hi));
        return this;
    }

    public Condition in(String property, Collection<?> values) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.IN).setValue(values));
        return this;
    }

    public Condition notIn(String property, Collection<?> values) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.NIN).setValue(values));
        return this;
    }

    public Condition isEmpty(String property) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.ISEMPTY));
        return this;
    }

    public Condition isNotEmpty(String property) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.ISNOTEMPTY));
        return this;
    }

    public Condition isNull(String property) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.ISNULL));
        return this;
    }

    public Condition isNotNull(String property) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.ISNOTNULL));
        return this;
    }

    public Condition isBoolean(String property, Object value) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.ISBOOLEAN).setValue(value));
        return this;
    }

    public Condition isTrue(String property) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.ISTRUE));
        return this;
    }

    public Condition isFalse(String property) {
        rules.add(new Rule().setProperty(property).setOp(Rule.Operator.ISFALSE));
        return this;
    }

    public Condition and() {
        if (rules.size() > 0) {
            groups.add(new RuleGroup()
                    .setRules(BeanConvertKit.convert(this.rules, Rule.class))
                    .setOp(RuleGroup.UnionOperator.AND));
            this.rules.clear();
        }
        return this;
    }

    public Condition or() {
        if (rules.size() > 0) {
            groups.add(new RuleGroup()
                    .setRules(BeanConvertKit.convert(this.rules, Rule.class))
                    .setOp(RuleGroup.UnionOperator.OR));
            this.rules.clear();
        }
        return this;
    }

    public Condition AND() {
        if (groups.size() > 0) {
            conditions.add(new Condition()
                    .setGroups(BeanConvertKit.convert(this.groups, RuleGroup.class))
                    .setRules(BeanConvertKit.convert(this.rules, Rule.class))
                    .setOp(RuleGroup.UnionOperator.AND));
            this.groups.clear();
            this.rules.clear();
        }
        return this;
    }

    public Condition OR() {
        if (groups.size() > 0) {
            conditions.add(new Condition()
                    .setGroups(BeanConvertKit.convert(this.groups, RuleGroup.class))
                    .setRules(BeanConvertKit.convert(this.rules, Rule.class))
                    .setOp(RuleGroup.UnionOperator.OR));
            this.groups.clear();
            this.rules.clear();
        }
        return this;
    }

    public Condition distinct() {
        distinct = true;
        return this;
    }

    public Condition groupBy(String property) {
        groupByNames.add(property);
        return this;
    }
}
