package com.gccloud.starter.common.filter;

import com.gccloud.starter.common.constant.GlobalConst;
import com.gccloud.starter.common.exception.GlobalException;
import com.gccloud.starter.common.vo.R;
import com.google.gson.Gson;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * 拦截 Oracle Application Server PL/SQL 未授权的 SQL 查询执行的漏洞
 * 这里最好是根据漏洞修改 Oracle Application Server 的规则配置
 *
 * @author liuchengbiao
 * @date 2021年12月09日09:16:32
 */
@Order(1)
@Component
@Slf4j
@ConditionalOnProperty(prefix = "gc.starter.component", name = "BugSecureFilter", havingValue = "BugSecureFilter", matchIfMissing = true)
public class BugSecureFilter implements Filter {

    @PostConstruct
    public void init() {
        log.info(GlobalConst.Console.LINE);
        log.info("启动漏洞修复拦截器");
        log.info(GlobalConst.Console.LINE);
    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) servletRequest;
        String servletPath = request.getServletPath();
        /**
         *  1、Oracle Application Server PL/SQL 未授权的 SQL 查询执行会有
         *  owa_util.signature、owa_util.listprint、owa_util.cellsprint 未经授权都可以访问的请求
         *  上面的请求可以直接操作oracle数据库，应该在 Oracle Application Server端进行规则配置，禁用上面的请求
         *  因为不清楚到底有多少个这样的请求，但是发现他们都是包含 /owa_util.
         *  所以，如果包含上面的就返回500
         *  2、Oracle 日志文件信息泄露
         *  地址： sqlnet.log、sqlnet.trc
         *  他们都有 sqlnet.
         */
        if (StringUtils.isBlank(servletPath)) {
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }
        try {
            if (servletPath.contains("/owa_util.")) {
                throw new GlobalException("禁用Oracle Application Server PL/SQL 未授权访问");
            }
            if (servletPath.contains("/sqlnet.")) {
                throw new GlobalException("禁用Oracle 日志文件信息泄露");
            }
            if (servletPath.endsWith(".lzma")) {
                throw new GlobalException("禁用临时文件下载");
            }
        } catch (Exception e) {
            log.error("疑似漏洞拦截: ", e);
            HttpServletResponse response = (HttpServletResponse) servletResponse;
            response.setHeader("Access-Control-Allow-Credentials", "true");
            response.setContentType("application/json;charset=UTF-8");
            response.setHeader("Access-Control-Allow-Origin", request.getHeader("Origin"));
            response.setStatus(500);
            String json = new Gson().toJson(R.error(GlobalConst.Response.Code.SERVER_ERROR, "非法访问"));
            response.getWriter().print(json);
            return;
        }
        filterChain.doFilter(servletRequest, servletResponse);
    }
}
