package com.gccloud.starter.common.filter;

import com.gccloud.starter.common.config.GlobalConfig;
import com.gccloud.starter.common.config.bean.Csrf;
import com.gccloud.starter.common.constant.GlobalConst;
import com.gccloud.starter.common.vo.R;
import com.google.gson.Gson;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;
import java.util.Map;

/**
 * 跨站点请求伪造-CSRF（Cross Site Request Forgery）：是一种网络攻击方式。
 * 思路：验证header中Referer是否是指定的值
 *
 * @author liuchengbiao
 * @date 2021年12月09日09:16:32
 */
@Order(1)
@Component
@Slf4j
@ConditionalOnProperty(prefix = "gc.starter.component", name = "CsrfSecureFilter", havingValue = "CsrfSecureFilter", matchIfMissing = true)
public class CsrfSecureFilter implements Filter {

    @Resource
    private GlobalConfig globalConfig;

    @PostConstruct
    public void init() {
        log.info(GlobalConst.Console.LINE);
        log.info("启动Csrf验证Referer拦截器");
        log.info(GlobalConst.Console.LINE);
    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) servletRequest;
        Csrf csrf = globalConfig.getCsrf();
        Map<Thread, StackTraceElement[]> allStackTraces = Thread.getAllStackTraces();
        String referer = request.getHeader("Referer");
        if (!csrf.getAllowedEmpty() && StringUtils.isBlank(referer)) {
            // 不允许空referer
            log.error("禁止 Referer 为空的访问");
            HttpServletResponse response = (HttpServletResponse) servletResponse;
            response.setHeader("Access-Control-Allow-Credentials", "true");
            response.setContentType("application/json;charset=UTF-8");
            response.setHeader("Access-Control-Allow-Origin", request.getHeader("Origin"));
            response.setStatus(500);
            String json = new Gson().toJson(R.error(GlobalConst.Response.Code.SERVER_ERROR, "非法访问"));
            response.getWriter().print(json);
            return;
        }
        if (StringUtils.isBlank(referer)) {
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }
        List<String> allowedReferers = csrf.getAllowedReferers();
        if (allowedReferers.size() == 0) {
            // 不进行验证
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }
        boolean allowed = false;
        for (String allowedReferer : allowedReferers) {
            if (referer.startsWith(allowedReferer)) {
                allowed = true;
                break;
            }
        }
        if (allowed) {
            // 允许访问
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }
        // 禁止访问
        log.error("禁止 Referer = {} 的访问", referer);
        HttpServletResponse response = (HttpServletResponse) servletResponse;
        response.setHeader("Access-Control-Allow-Credentials", "true");
        response.setContentType("application/json;charset=UTF-8");
        response.setHeader("Access-Control-Allow-Origin", request.getHeader("Origin"));
        response.setStatus(500);
        String json = new Gson().toJson(R.error(GlobalConst.Response.Code.SERVER_ERROR, "非法访问"));
        response.getWriter().print(json);
    }
}
