package com.gccloud.starter.common.module.monitor.aop;

import com.gccloud.starter.common.constant.GlobalConst;
import com.gccloud.starter.common.exception.GlobalException;
import com.gccloud.starter.common.utils.LambdaUtils;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.base.Stopwatch;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.annotation.Order;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.PostConstruct;
import java.lang.reflect.Method;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 接口监控
 * 拦截所有Controller方法
 * 记录调用的次数、耗时、异常次数等信息
 * 便于出现问题时根据相关指标快速定位可能是什么出现了问题
 *
 * @author liuchengbiao
 * @date 2020-06-19 10:19
 */
@Aspect
@Component
@RestController
@Slf4j
@Api(tags = "接口监控")
@ConditionalOnProperty(prefix = "gc.starter.component", name = "SysApiMonitorAspect", havingValue = "SysApiMonitorAspect", matchIfMissing = true)
public class SysApiMonitorAspect {

    private static final Cache<String, MethodMonitorDTO> CACHE = Caffeine.newBuilder().build();

    @PostConstruct
    public void init() {
        log.info(GlobalConst.Console.LINE);
        log.info("初始化监控切面，监控所有com包名的Controller或org包名的Controller");
        log.info(GlobalConst.Console.LINE);
    }

    /**
     * 拦截所有 controller 命名不符合要求的将不进行拦截
     */
    @Pointcut("execution(*  com..*Controller.*(..)) || execution(*  org..*Controller.*(..))")
    @Order(10)
    public void pointCut() {
    }

    @Around("pointCut()")
    public Object logPointCut(ProceedingJoinPoint joinPoint) throws Throwable {
        Stopwatch stopwatch = Stopwatch.createStarted();
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        Method method = signature.getMethod();
        String name = method.getDeclaringClass().getName() + "." + method.getName();
        boolean exception = false;
        try {
            return joinPoint.proceed();
        } catch (Exception e) {
            exception = true;
            log.error(ExceptionUtils.getStackTrace(e));
            throw e;
        } finally {
            MethodMonitorDTO monitorDTO = CACHE.get(name, (k) -> {
                MethodMonitorDTO monitor = new MethodMonitorDTO();
                monitor.setMethodName(name);
                return monitor;
            });
            monitorDTO.getExeCount().incrementAndGet();
            long elapsed = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            monitorDTO.getTotalElapsedTime().addAndGet(elapsed);
            if (exception) {
                monitorDTO.getExceptionCount().incrementAndGet();
            }
            // 耗时时间段统计
            if (elapsed >= 5000) {
                monitorDTO.getGe5sCount().incrementAndGet();
            } else if (elapsed >= 4000) {
                monitorDTO.getGe4sCount().incrementAndGet();
            } else if (elapsed >= 3000) {
                monitorDTO.getGe3sCount().incrementAndGet();
            } else if (elapsed >= 2000) {
                monitorDTO.getGe2sCount().incrementAndGet();
            } else if (elapsed >= 1000) {
                monitorDTO.getGe1sCount().incrementAndGet();
            }
        }
    }


    @ApiOperation(value = "接口耗时", position = 10, notes = "查询接口耗时", produces = MediaType.APPLICATION_JSON_VALUE)
    @ApiImplicitParams({@ApiImplicitParam(name = "sortName", value = "排序名称(evgElapsedTime: 平均耗时," + " totalElapsedTime: 总耗时," + " exceptionCount: 异常次数," + " ge5sCount: 大于5秒的次数," + " ge4sCount: 大于4秒的次数," + " ge3sCount: 大于3秒的次数," + " ge2sCount: 大于2秒的次数," + " ge1sCount: 大于1秒的次数," + " exeCount: 调用次数)", paramType = "path", required = true, dataType = "string"), @ApiImplicitParam(name = "sortMethod", value = "排序(asc: 升序, desc: 降序)", paramType = "path", dataType = "string")})
    @GetMapping("/getMethodMonitorList/{sortName}/{sortMethod}")
    public List<MethodMonitorDTO> getMethodMonitorList(@PathVariable(value = "sortName") String sortName, @PathVariable("sortMethod") String sortMethod) {
        List<MethodMonitorDTO> list = CACHE.asMap().values().stream().collect(Collectors.toList());
        for (MethodMonitorDTO methodMonitorDTO : list) {
            long evg = methodMonitorDTO.getTotalElapsedTime().get() / methodMonitorDTO.getExeCount().get();
            methodMonitorDTO.setAvgElapsedTime(evg);
        }
        if (LambdaUtils.getFieldName(MethodMonitorDTO::getAvgElapsedTime).equals(sortName)) {
            list.sort((a, b) -> {
                if (MethodMonitorDTO.SORT_ASC.equals(sortMethod)) {
                    return (a.getAvgElapsedTime() - b.getAvgElapsedTime()) > 0 ? 1 : -1;
                }
                return (a.getAvgElapsedTime() - b.getAvgElapsedTime()) > 0 ? -1 : 1;
            });
            return list;
        }
        if (LambdaUtils.getFieldName(MethodMonitorDTO::getTotalElapsedTime).equals(sortName)) {
            list.sort((a, b) -> {
                if (MethodMonitorDTO.SORT_ASC.equals(sortMethod)) {
                    return (a.getTotalElapsedTime().get() - b.getTotalElapsedTime().get()) > 0 ? 1 : -1;
                }
                return (a.getTotalElapsedTime().get() - b.getTotalElapsedTime().get()) > 0 ? -1 : 1;
            });
            return list;
        }
        if (LambdaUtils.getFieldName(MethodMonitorDTO::getExeCount).equals(sortName)) {
            list.sort((a, b) -> {
                if (MethodMonitorDTO.SORT_ASC.equals(sortMethod)) {
                    return (a.getExeCount().get() - b.getExeCount().get()) > 0 ? 1 : -1;
                }
                return (a.getExeCount().get() - b.getExeCount().get()) > 0 ? -1 : 1;
            });
            return list;
        }
        if (LambdaUtils.getFieldName(MethodMonitorDTO::getGe5sCount).equals(sortName)) {
            list.sort((a, b) -> {
                if (MethodMonitorDTO.SORT_ASC.equals(sortMethod)) {
                    return (a.getGe5sCount().get() - b.getGe5sCount().get()) > 0 ? 1 : -1;
                }
                return (a.getGe5sCount().get() - b.getGe5sCount().get()) > 0 ? -1 : 1;
            });
            return list;
        }
        if (LambdaUtils.getFieldName(MethodMonitorDTO::getGe4sCount).equals(sortName)) {
            list.sort((a, b) -> {
                if (MethodMonitorDTO.SORT_ASC.equals(sortMethod)) {
                    return (a.getGe4sCount().get() - b.getGe4sCount().get()) > 0 ? 1 : -1;
                }
                return (a.getGe4sCount().get() - b.getGe4sCount().get()) > 0 ? -1 : 1;
            });
            return list;
        }
        if (LambdaUtils.getFieldName(MethodMonitorDTO::getGe3sCount).equals(sortName)) {
            list.sort((a, b) -> {
                if (MethodMonitorDTO.SORT_ASC.equals(sortMethod)) {
                    return (a.getGe3sCount().get() - b.getGe3sCount().get()) > 0 ? 1 : -1;
                }
                return (a.getGe3sCount().get() - b.getGe3sCount().get()) > 0 ? -1 : 1;
            });
            return list;
        }
        if (LambdaUtils.getFieldName(MethodMonitorDTO::getGe2sCount).equals(sortName)) {
            list.sort((a, b) -> {
                if (MethodMonitorDTO.SORT_ASC.equals(sortMethod)) {
                    return (a.getGe2sCount().get() - b.getGe2sCount().get()) > 0 ? 1 : -1;
                }
                return (a.getGe2sCount().get() - b.getGe2sCount().get()) > 0 ? -1 : 1;
            });
            return list;
        }
        if (LambdaUtils.getFieldName(MethodMonitorDTO::getGe1sCount).equals(sortName)) {
            list.sort((a, b) -> {
                if (MethodMonitorDTO.SORT_ASC.equals(sortMethod)) {
                    return (a.getGe1sCount().get() - b.getGe1sCount().get()) > 0 ? 1 : -1;
                }
                return (a.getGe1sCount().get() - b.getGe1sCount().get()) > 0 ? -1 : 1;
            });
            return list;
        }
        throw new GlobalException("参数非法");
    }
}
