package com.gccloud.starter.common.module.monitor.service.impl;

import cn.hutool.core.net.NetUtil;
import cn.hutool.core.util.NumberUtil;
import com.gccloud.starter.common.module.monitor.service.ISysMonitorService;
import com.gccloud.starter.common.module.monitor.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.stereotype.Service;
import oshi.SystemInfo;
import oshi.hardware.CentralProcessor;
import oshi.hardware.GlobalMemory;
import oshi.hardware.HardwareAbstractionLayer;
import oshi.software.os.FileSystem;
import oshi.software.os.OSFileStore;
import oshi.software.os.OperatingSystem;
import oshi.util.Util;

import java.net.InetAddress;
import java.util.List;
import java.util.Properties;

/**
 * @author liuchengbiao
 * @date 2021/9/23 11:58 上午
 */
@Service
@Slf4j
public class SysMonitorServiceImpl implements ISysMonitorService {

    private static final int OSHI_WAIT_SECOND = 1000;

    /**
     * 刷新数据
     */
    @Override
    public MonitorVo getData() {
        MonitorVo monitorVo = new MonitorVo();
        SystemInfo si = new SystemInfo();
        HardwareAbstractionLayer hal = si.getHardware();
        setCpuInfo(monitorVo, hal.getProcessor());
        setMemInfo(monitorVo, hal.getMemory());
        setSysInfo(monitorVo);
        setJvmInfo(monitorVo);
        setFileSystem(monitorVo, si.getOperatingSystem());
        return monitorVo;
    }

    /**
     * 设置CPU信息
     */
    private void setCpuInfo(MonitorVo monitorVo, CentralProcessor processor) {
        // CPU信息
        long[] prevTicks = processor.getSystemCpuLoadTicks();
        Util.sleep(OSHI_WAIT_SECOND);
        long[] ticks = processor.getSystemCpuLoadTicks();
        long nice = ticks[CentralProcessor.TickType.NICE.getIndex()] - prevTicks[CentralProcessor.TickType.NICE.getIndex()];
        long irq = ticks[CentralProcessor.TickType.IRQ.getIndex()] - prevTicks[CentralProcessor.TickType.IRQ.getIndex()];
        long softirq = ticks[CentralProcessor.TickType.SOFTIRQ.getIndex()] - prevTicks[CentralProcessor.TickType.SOFTIRQ.getIndex()];
        long steal = ticks[CentralProcessor.TickType.STEAL.getIndex()] - prevTicks[CentralProcessor.TickType.STEAL.getIndex()];
        long cSys = ticks[CentralProcessor.TickType.SYSTEM.getIndex()] - prevTicks[CentralProcessor.TickType.SYSTEM.getIndex()];
        long user = ticks[CentralProcessor.TickType.USER.getIndex()] - prevTicks[CentralProcessor.TickType.USER.getIndex()];
        long iowait = ticks[CentralProcessor.TickType.IOWAIT.getIndex()] - prevTicks[CentralProcessor.TickType.IOWAIT.getIndex()];
        long idle = ticks[CentralProcessor.TickType.IDLE.getIndex()] - prevTicks[CentralProcessor.TickType.IDLE.getIndex()];
        long totalCpu = user + nice + cSys + idle + iowait + irq + softirq + steal;
        CpuVo cpu = monitorVo.getCpu();
        cpu.setCpuNum(processor.getLogicalProcessorCount());
        cpu.setTotal(totalCpu);
        cpu.setSys(cSys);
        cpu.setUsed(user);
        cpu.setWait(iowait);
        cpu.setFree(idle);
    }

    /**
     * 设置内存信息
     */
    private void setMemInfo(MonitorVo monitorVo, GlobalMemory memory) {
        MemoryVo mem = monitorVo.getMemory();
        mem.setTotal(memory.getTotal());
        mem.setUsed(memory.getTotal() - memory.getAvailable());
        mem.setFree(memory.getAvailable());
    }

    /**
     * 设置服务器信息
     */
    private void setSysInfo(MonitorVo monitorVo) {
        SystemVo sys = monitorVo.getSystem();
        Properties props = System.getProperties();
        try {
            InetAddress addr = InetAddress.getLocalHost();
            // 获取本机计算机名称
            String hostName = addr.getHostName();
            sys.setComputerName(hostName);
        } catch (Exception e) {
            log.error(ExceptionUtils.getStackTrace(e));
            sys.setComputerName("未知");
        }
        sys.setComputerIp(NetUtil.getLocalhostStr());
        sys.setOsName(props.getProperty("os.name"));
        sys.setOsArch(props.getProperty("os.arch"));
        sys.setUserDir(props.getProperty("user.dir"));
    }

    /**
     * 设置Java虚拟机
     */
    private void setJvmInfo(MonitorVo monitorVo) {
        JvmVo jvm = monitorVo.getJvm();
        Properties props = System.getProperties();
        jvm.setTotal(Runtime.getRuntime().totalMemory());
        jvm.setMax(Runtime.getRuntime().maxMemory());
        jvm.setFree(Runtime.getRuntime().freeMemory());
        jvm.setVersion(props.getProperty("java.version"));
        jvm.setHome(props.getProperty("java.home"));
    }

    /**
     * 设置磁盘信息
     */
    private void setFileSystem(MonitorVo monitorVo, OperatingSystem os) {
        List<FileSystemVo> sysFiles = monitorVo.getFileSystemList();
        FileSystem fileSystem = os.getFileSystem();
        List<OSFileStore> fsList = fileSystem.getFileStores();
        for (OSFileStore fs : fsList) {
            long free = fs.getUsableSpace();
            long total = fs.getTotalSpace();
            long used = total - free;
            FileSystemVo sysFile = new FileSystemVo();
            sysFile.setDirName(fs.getMount());
            sysFile.setSysTypeName(fs.getType());
            sysFile.setTypeName(fs.getName());
            sysFile.setTotal(convertFileSize(total));
            sysFile.setFree(convertFileSize(free));
            sysFile.setUsed(convertFileSize(used));
            //sysFile.setUsage(NumberUtil.round(NumberUtil.mul(used, total, 4), 2).doubleValue());
            sysFile.setUsage(NumberUtil.round(used * 1.0 / total * 100, 2).doubleValue());
            sysFiles.add(sysFile);
        }
    }

    /**
     * 字节转换
     *
     * @param size 字节大小
     * @return 转换后值
     */
    public String convertFileSize(long size) {
        long kb = 1024;
        long mb = kb * 1024;
        long gb = mb * 1024;
        if (size >= gb) {
            return String.format("%.1f GB", (float) size / gb);
        } else if (size >= mb) {
            float f = (float) size / mb;
            return String.format(f > 100 ? "%.0f MB" : "%.1f MB", f);
        } else if (size >= kb) {
            float f = (float) size / kb;
            return String.format(f > 100 ? "%.0f KB" : "%.1f KB", f);
        } else {
            return String.format("%d B", size);
        }
    }

    public static void main(String[] args) throws Exception {
        long total = 250790436864L, used = 222992535552L;
        System.out.println(used * 1.0 / total);
        System.out.println(NumberUtil.round(used * 1.0 / total * 100, 2));
        System.out.println(NumberUtil.mul(used, total, 2));
        System.out.println(NumberUtil.round(NumberUtil.mul(used, total, 2), 2).doubleValue());
    }
}
