package com.gccloud.starter.common.module.secure.service.impl;

import com.gccloud.starter.common.constant.GlobalConst;
import com.gccloud.starter.common.module.secure.service.IWeakPwdService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.shiro.crypto.hash.Sha256Hash;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * 弱密码库
 *
 * @author liuchengbiao
 * @date 2021/6/7 10:32 上午
 */
@Component
@Slf4j
@ConditionalOnProperty(prefix = "gc.starter.component", name = "IWeakPwdService", havingValue = "WeakPwdServiceImpl", matchIfMissing = true)
public class WeakPwdServiceImpl implements IWeakPwdService {
    protected static final Set<String> LIB = new HashSet<>(17800);
    /**
     * 弱密码文件
     */
    public static final String PWD_FILE_NAME = "weakPwdLib.txt";
    /**
     * 弱密码文件目录
     */
    public static final String PWD_FILE_PATH = "src/main/resources";

    @PostConstruct
    public void init() {
        log.info(GlobalConst.Console.LINE);
        log.info("开始加载弱密码文件:{}/{},后面不允许进行弱密码访问、设置", PWD_FILE_PATH, PWD_FILE_NAME);
        try (InputStream inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(PWD_FILE_NAME);) {
            if (inputStream == null) {
                log.error("丢失弱密码库文件 {} 请在 {} 目录下添加，以明文形式，每行一个", PWD_FILE_NAME, PWD_FILE_PATH);
                return;
            }
            List<String> lineList = IOUtils.readLines(inputStream, StandardCharsets.UTF_8.name());
            for (String pwd : lineList) {
                if (StringUtils.isBlank(pwd)) {
                    continue;
                }
                // 明文转为密文
                String encryptPassword = new Sha256Hash(new Sha256Hash(pwd).toHex() + pwd).toHex();
                LIB.add(encryptPassword);
            }
            log.info("共加载弱密码:{} 条", LIB.size());
        } catch (IOException e) {
            log.error(ExceptionUtils.getStackTrace(e));
        } finally {
            log.info(GlobalConst.Console.LINE);
        }
    }

    /**
     * 是否在弱密码库中存在
     *
     * @param encryptPassword
     * @return
     */
    public boolean exist(String encryptPassword) {
        if (StringUtils.isBlank(encryptPassword)) {
            // 兼容不通过密码进行访问情况
            return false;
        }
        return LIB.contains(encryptPassword);
    }
}


