package com.gccloud.starter.common.utils;

import com.gccloud.starter.common.entity.SysMenuEntity;
import com.gccloud.starter.common.exception.GlobalException;
import com.gccloud.starter.common.vo.CurrentUserBase;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.shiro.SecurityUtils;
import org.apache.shiro.subject.Subject;

import java.util.concurrent.TimeUnit;
import java.util.function.Function;

@Slf4j
public class UserUtils {

    /**
     * 匿名接口用户上下文
     */
    private static final ThreadLocal<CurrentUserBase> ANON_USER_CONTEXT = new ThreadLocal<>();
    /**
     * 数据规则编码缓存
     */
    private static final ThreadLocal<String> DATA_RULE_CODE_CONTEXT = new ThreadLocal<>();
    /**
     * 缓存一天
     */
    private static final Cache<String, CurrentUserBase> CURRENT_USER_CACHE = Caffeine.newBuilder()
            // 缓存一天
            .expireAfterAccess(1, TimeUnit.DAYS)
            // 设置最大记录数
            .maximumSize(10000).build();

    public UserUtils() {
        throw new IllegalStateException("不允许创建");
    }

    /**
     * 设置当前接口数据规则编码
     *
     * @param dataRuleCode
     */
    public static void putCurrentDataRuleCode(String dataRuleCode) {
        DATA_RULE_CODE_CONTEXT.set(dataRuleCode);
    }

    /**
     * 删除当前接口数据规则编码，防止内存泄漏
     */
    public static void removeCurrentDataRuleCode() {
        DATA_RULE_CODE_CONTEXT.remove();
    }

    /**
     * 获取当前接口数据规则编码
     *
     * @return
     */
    public static String getCurrentDataRuleCode() {
        return DATA_RULE_CODE_CONTEXT.get();
    }

    /**
     * 获取当前接口数据规则
     *
     * @param dataRuleCode
     * @return
     */
    public static SysMenuEntity getCurrentDataRule(String dataRuleCode) {
        CurrentUserBase currentUser = getCurrentUser();
        return currentUser.getDataRuleMap().get(dataRuleCode);
    }

    /**
     * 获取用户
     *
     * @param username
     * @param callBack
     * @return
     */
    public static CurrentUserBase getUser(String username, Function<String, CurrentUserBase> callBack) {
        CurrentUserBase currentUser = CURRENT_USER_CACHE.get(username, callBack);
        return currentUser;
    }

    /**
     * 清空缓存用户
     */
    public static void invalidateAllUser() {
        CURRENT_USER_CACHE.invalidateAll();
    }

    /**
     * 设置上下文用户
     *
     * @param user
     */
    public static void putAnonUser(CurrentUserBase user) {
        ANON_USER_CONTEXT.set(user);
    }

    /**
     * 移除用户；防止内存溢出
     */
    public static void removeAnonUser() {
        ANON_USER_CONTEXT.remove();
    }

    /**
     * 返回当前用户的租户ID
     *
     * @return
     */
    public static String getTenantId() {
        return getCurrentUser().getTenantId();
    }

    public static String tryGetTenantId() {
        CurrentUserBase currentUser = getCurrentUser();
        return currentUser.getTenantId();
    }

    /**
     * 返回当前用户
     *
     * @return
     */
    public static CurrentUserBase getCurrentUser() {
        return getCurrentUser(true);
    }

    /**
     * 获取当前用户
     *
     * @param nullThrowException 不存在用户是否抛出异常
     * @return
     */
    public static CurrentUserBase getCurrentUser(boolean nullThrowException) {
        Subject subject = null;
        try {
            subject = SecurityUtils.getSubject();
        } catch (Exception e) {
            if (nullThrowException){
                log.error(ExceptionUtils.getStackTrace(e));
            }else{
                // 可以不用排除异常信息
                log.debug(ExceptionUtils.getStackTrace(e));
            }
        }
        if (subject == null) {
            // 从上下文中获取
            CurrentUserBase user = ANON_USER_CONTEXT.get();
            if (user != null) {
                return user;
            }
            if (nullThrowException) {
                throw new GlobalException("未获取到用户信息");
            }
            return null;
        }
        Object principal = subject.getPrincipal();
        if (principal == null) {
            // 从上下文中获取
            CurrentUserBase user = ANON_USER_CONTEXT.get();
            if (user != null) {
                return user;
            }
            if (nullThrowException) {
                throw new GlobalException("未获取到用户信息");
            }
            return null;
        }
        return (CurrentUserBase) principal;
    }

    /**
     * 获取当前登录的用户的ID
     * 不存在时返回 -1
     *
     * @return
     */
    public static String getCurrentUserId() {
        CurrentUserBase currentUser = getCurrentUser();
        return currentUser.getId();
    }

    /**
     * 获取当前登录的用户名
     *
     * @return
     */
    public static String getCurrentUserName() {
        CurrentUserBase currentUser = getCurrentUser();
        return currentUser.getName();
    }
}
